data "external" "deploy_db_archive" {
  program = ["bash", "build.sh", "deploy_db_from_snapshot_dest_aurora"]
}

data "aws_ssm_parameter" "db_password" {
  provider = aws.twitch-grafana-dev
  name     = "/twitch-grafana-dev/db-password-plain.txt"
}

resource "aws_lambda_function" "aurora_lambdas_deploy" {
  provider         = aws.twitch-grafana-dev
  function_name    = "deploy-prod-aurora-snapshot-db"
  filename         = data.external.deploy_db_archive.result.path
  source_code_hash = data.external.deploy_db_archive.result.hash
  role             = aws_iam_role.aurora_lambdas.arn
  handler          = "lambda_handler.lambda_handler"
  runtime          = "python3.6"
  timeout          = 120
  vpc_config {
    security_group_ids = var.vpc_security_group_ids
    subnet_ids         = var.subnet_ids
  }
  environment {
    variables = {
      LAMBDA           = "true"
      COPY_TO_DB       = "sql.grafana-dev.twitch.a2z.com"
      PGPASSWORD       = data.aws_ssm_parameter.db_password.value
      DB_SUBNET_NAME   = "grafana"
      LOG_LEVEL        = "INFO"
      SNAPSHOT_PATTERN = "grafana-aurora-cluster-production" # only deploy prod snaps.
    }
  }
  reserved_concurrent_executions = 1
}

resource "aws_cloudwatch_event_rule" "every_day_at_6" {
  provider    = aws.twitch-grafana-dev
  name        = "deploy-aurora-prod-db-daily"
  description = "Deploys a fresh production aurora db every day."

  # schedule_expression = "cron(45 3 * * ? *)" # 8:45 PM PST
  schedule_expression = "cron(30 13 * * ? *)" # 6:30 AM PST
}

resource "aws_sfn_state_machine" "aurora_lambdas_deploy" {
  provider   = aws.twitch-grafana-dev
  name       = "deploy-prod-aurora-snapshot-db"
  role_arn   = aws_iam_role.aurora_lambdas.arn
  definition = <<EOF
{
  "Comment": "Deploy for Aurora prod database",
  "StartAt": "DeployProdDatabase",
  "States": {
    "DeployProdDatabase": {
      "Type":     "Task",
      "Resource": "${aws_lambda_function.aurora_lambdas_deploy.arn}",
      "Retry":    [
        {
          "ErrorEquals":     ["SnapshotToolException"],
          "IntervalSeconds": 400,
          "MaxAttempts":     5,
          "BackoffRate":     1
        },
        {
          "ErrorEquals":     ["States.ALL"],
          "IntervalSeconds": 120,
          "MaxAttempts":     15,
          "BackoffRate":     1
       }
     ],
     "End": true
    }
  }
}
EOF

}

resource "aws_cloudwatch_event_target" "aurora_lambdas_deploy" {
  provider = aws.twitch-grafana-dev
  rule     = aws_cloudwatch_event_rule.every_day_at_6.name
  arn      = aws_sfn_state_machine.aurora_lambdas_deploy.id
  role_arn = aws_iam_role.aurora_lambdas.arn
}

