'''
Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License").
You may not use this file except in compliance with the License.
A copy of the License is located at

    http://aws.amazon.com/apache2.0/

or in the "license" file accompanying this file.
This file is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
either express or implied. See the License for
the specific language governing permissions
and limitations under the License.
'''

# deploy_db_from_snapshot_dest_aurora
# Deploys a database from the latest snapshot that matches PATTERN.
# Set PATTERN to a regex that matches your Aurora cluster identifiers (by default: <instance_name>-cluster)
# Set DEST_REGION to the destination AWS region
import boto3
import os
import logging
from subprocess import Popen, PIPE, STDOUT
from shlex import split
import snapshots_tool_utils as util

# Initialize everything
PREFIX = 'ss-tool-'
LOGLEVEL = os.getenv('LOG_LEVEL', 'INFO').strip()
PATTERN = os.getenv('SNAPSHOT_PATTERN', 'grafana-aurora-cluster-production')
DESTINATION_REGION = os.getenv('DEST_REGION', os.getenv('AWS_DEFAULT_REGION', '')).strip()
# deeper config
COPY_TO_DB = os.getenv('COPY_TO_DB', "sql.grafana-dev.twitch.a2z.com").strip()
# passed in from terraform
DB_SUBNET_NAME = os.getenv('DB_SUBNET_NAME', 'grafana').strip()

logging.basicConfig()

logger = logging.getLogger()
logger.setLevel(LOGLEVEL.upper())


def lambda_handler(event, context):
    client = boto3.client('rds', region_name=DESTINATION_REGION)
    snapshots = util.paginate_api_call(client, 'describe_db_cluster_snapshots', 'DBClusterSnapshots')
    snapshots = util.get_own_snapshots_dest(PATTERN, snapshots)
    clusters = util.paginate_api_call(client, 'describe_db_clusters', 'DBClusters')
    clusters = util.filter_clusters(PREFIX, clusters)

    snap_len = len(snapshots)
    for i, var in enumerate(snapshots):
        # Assume the last snapshot is the latest? ok..
        if i == snap_len - 1:
            latest_snapshot = var
    logger.info("Latest Snapshot: {0} (Total: {1})".format(latest_snapshot, snap_len))

    # Check if we already restored the latest snapshot.
    for db_cluster in clusters:
        if db_cluster['DBClusterIdentifier'] == PREFIX + latest_snapshot:
            logger.error("Latest Snapshot '{0}' already exists as a Cluster!".format(latest_snapshot))
            if db_cluster['Status'] == 'creating':
                log_message = ("Snapshot still restoring. Retrying to finish database copy." +
                               " Cluster: {0}".format(PREFIX + latest_snapshot))
                raise util.SnapshotToolException(log_message)
            copy_database(db_cluster['Endpoint'], db_cluster['DatabaseName'], db_cluster['MasterUsername'])
            return

    logger.error("Cluster missing, restoring snapshot: {0}".format(latest_snapshot))
    client.restore_db_cluster_from_snapshot(
        DBClusterIdentifier=PREFIX + latest_snapshot,
        SnapshotIdentifier=latest_snapshot,
        Engine='aurora-postgresql',
        DBSubnetGroupName=DB_SUBNET_NAME,
        EngineMode='serverless',
    )
    # check response?

    log_message = ("Snapshot restoring. Retrying to finish database copy." +
                   " New Cluster: {0}".format(PREFIX + latest_snapshot))
    raise util.SnapshotToolException(log_message)


def copy_database(endpoint, db_name, db_username):
    """ This procedures copies a database using raw psql commands.
        No input validation performed! """
    logger.info("Copying DB with pg_dump: {0}:{2} -> {1}:{2} (user: {3})".format(
        endpoint, COPY_TO_DB, db_name, db_username))
    if COPY_TO_DB == "":
        raise util.SnapshotToolException("Restoration DB empty! Fix this ENV variable!")

    with open('/tmp/temp.txt', 'w') as output:
        dump = Popen(split("./pg_dump -c -h {0} -U {1} {2}".format(
            endpoint, db_username, db_name)), stdout=PIPE, stderr=output)
        copy = Popen(split("./psql -h {0} -U {1} {2}".format(
            COPY_TO_DB, db_username, db_name)), stdin=dump.stdout, stdout=PIPE, stderr=STDOUT)
        dump.stdout.close()
        out = copy.communicate()[0]
        dump.wait()
        logger.info(out)

    with open('/tmp/temp.txt', 'r') as output:
        dout = output.read()  # read in the log file.

    if dump.returncode != 0:
        log_message = "Dump Failed! (exit code:{0}) {1}".format(dump.returncode, dout)
        raise util.SnapshotToolException(log_message)
    else:
        logger.debug(dout)
    if copy.returncode != 0:
        log_message = "Copy Failed! (exit code:{0}) {1}".format(copy.returncode, out)
        raise util.SnapshotToolException(log_message)


if __name__ == '__main__':
    lambda_handler(None, None)
