### lb

resource "aws_alb" "grafana_server" {
  name            = "ecs-grafana-server"
  internal        = true
  security_groups = var.vpc_security_group_ids
  subnets         = var.subnet_ids
}

resource "aws_alb_target_group" "grafana_server" {
  name        = "grafana-server"
  port        = 3000
  protocol    = "HTTP"
  vpc_id      = var.vpc_id
  target_type = "ip"

  health_check {
    healthy_threshold   = 5
    unhealthy_threshold = 3
    timeout             = 5
    interval            = 10
    path                = "/logout"
    matcher             = "200,302"
  }
}

# Redirect all traffic from the lb to the target group
# certificate_arn = grafana-dev.twitch.a2z.com
resource "aws_alb_listener" "grafana_server" {
  load_balancer_arn = aws_alb.grafana_server.arn
  port              = "443"
  protocol          = "HTTPS"
  certificate_arn   = "arn:aws:acm:us-west-2:963768028156:certificate/85e965ca-5b7d-4e27-8ca7-7fd6914719c7"

  default_action {
    target_group_arn = aws_alb_target_group.grafana_server.id
    type             = "forward"
  }
}

# Redirect all traffic from port 80 to 443.
resource "aws_alb_listener" "grafana_server_redirect" {
  load_balancer_arn = aws_alb.grafana_server.arn
  port              = "80"
  protocol          = "HTTP"

  default_action {
    type = "redirect"

    redirect {
      port        = "443"
      protocol    = "HTTPS"
      status_code = "HTTP_301"
    }
  }
}

# We can use the privatelink to give other AWS accounts (and VPN) access to Grafana.
module "privatelink" {
  source      = "git::git+ssh://git@git.xarth.tv/terraform-modules/privatelink-alb.git?ref=v0.0.6"
  account     = "twitch-grafana-dev"
  service     = "ecs-grafana"
  environment = var.environment
  dns_name    = aws_alb.grafana_server.dns_name
  vpc         = var.vpc_id
  subnets     = var.subnet_ids
}

# DNS for ALB - allows image-renderer to call back in, locally.
resource "aws_route53_record" "lb_dns" {
  zone_id = "Z3UD4SDI868JL2" # grafana-dev.twitch.a2z.com.
  name    = ""               # grafana-dev.twitch.a2z.com.
  type    = "A"

  alias {
    name                   = aws_alb.grafana_server.dns_name
    zone_id                = aws_alb.grafana_server.zone_id
    evaluate_target_health = false
  }
}

# output
output "alb_name" {
  value = aws_alb.grafana_server.dns_name
}

output "vpc_endpoint_name" {
  value = module.privatelink.vpc_endpoint_name
}
