resource "aws_appautoscaling_target" "grafana_server" {
  service_namespace  = "ecs"
  resource_id        = "service/${aws_ecs_cluster.grafana_server.name}/${aws_ecs_service.grafana_server.name}"
  scalable_dimension = "ecs:service:DesiredCount"
  max_capacity       = var.service_max_instances
  min_capacity       = var.service_min_instances
}

resource "aws_ecs_cluster" "grafana_server" {
  name = "grafana-server"
}

resource "aws_ecs_task_definition" "grafana_server" {
  family                   = "grafana-server-${var.environment}"
  network_mode             = "awsvpc"
  requires_compatibilities = ["FARGATE"]
  cpu                      = var.fargate_cpu
  memory                   = var.fargate_memory
  execution_role_arn       = aws_iam_role.ecsTaskExecutionRole.arn
  task_role_arn            = aws_iam_role.ecsTaskExecutionRole.arn

  container_definitions = <<DEFINITION
[
  {
    "name": "grafana-server-${var.environment}",
    "cpu": ${var.fargate_cpu},
    "image": "${var.service_image}",
    "memory": ${var.fargate_memory},
    "name": "grafana-server-${var.environment}",
    "essential": true,
    "networkMode": "awsvpc",
    "secrets": [
      {
        "name": "DB_PASSWORD",
        "valueFrom": "arn:aws:ssm:us-west-2:109561199089:parameter/twitch-grafana-prod/db_password.txt"
      },
      {
        "name": "SES_PASSWORD",
        "valueFrom": "arn:aws:ssm:us-west-2:109561199089:parameter/twitch-grafana-prod/ses_password.txt"
      },
      {
        "name": "EXTRA_CONFIG",
        "valueFrom": "arn:aws:ssm:us-west-2:109561199089:parameter/twitch-grafana-prod/extra_config.ini"
      },
      {
        "name": "GRAFANA_ENTERPRISE_LICENSE_CONTENT",
        "valueFrom": "arn:aws:secretsmanager:us-west-2:109561199089:secret:grafana-enterprise-license-jwt-prod-oBO5La"
      }
    ],
    "portMappings": [
      {
        "containerPort": ${var.service_port}
      }
    ],
    "environment": [
      {
        "name": "GF_RENDERING_SERVER_URL",
        "value": "http://${aws_alb.grafana_image_renderer.dns_name}:8081/render"
      },
      {
        "name": "GF_RENDERING_CALLBACK_URL",
        "value": "https://${aws_route53_record.lb_dns.fqdn}/"
      },
      {
        "name": "GF_SERVER_DOMAIN",
        "value": "${aws_route53_record.lb_dns.fqdn}"
      },
      {
        "name": "GF_SERVER_ROOT_URL",
        "value": "https://${aws_route53_record.lb_dns.fqdn}"
      },
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      },
      {
        "name": "AWS_STS_REGIONAL_ENDPOINTS",
        "value": "regional"
      }
    ],
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "/fargate/service/grafana-server-${var.environment}",
        "awslogs-region": "us-west-2",
        "awslogs-stream-prefix": "ecs"
      }
    }
  }
]
DEFINITION
}

resource "aws_ecs_service" "grafana_server" {
  name             = "grafana-server-${var.environment}"
  cluster          = aws_ecs_cluster.grafana_server.id
  task_definition  = aws_ecs_task_definition.grafana_server.arn
  desired_count    = var.service_min_instances
  launch_type      = "FARGATE"
  platform_version = "1.4.0"

  network_configuration {
    security_groups = var.vpc_security_group_ids
    subnets         = var.subnet_ids
  }

  load_balancer {
    target_group_arn = aws_alb_target_group.grafana_server.arn
    container_name   = "grafana-server-${var.environment}"
    container_port   = var.service_port
  }

  depends_on = [
    aws_alb_listener.grafana_server,
  ]

  # In most cases we won't want our terraform run to modify the task definition after the initial run
  # Things like the image will be modified by deploys etc potentially, and we dont want to run terraform every time
  lifecycle {
    ignore_changes = [task_definition, desired_count]
  }
}

# For task executions
resource "aws_iam_role" "ecsTaskExecutionRole" {
  name = "grafana-${var.environment}"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": [
          "ec2.amazonaws.com",
          "ecs-tasks.amazonaws.com",
          "events.amazonaws.com"
        ]
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

# Allow the ECS tasks to grab the parameters in SSM.
resource "aws_iam_policy" "grafana_server_policy" {
  name        = "grafana-server-policy"
  description = "Allows access to SSM parameters and cloudwatch data in other accounts."

  policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "ssm:DescribeParameters"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "ssm:GetParameters"
            ],
            "Resource": "arn:aws:ssm:us-west-2:109561199089:parameter/twitch-grafana-prod/*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "secretsmanager:GetSecretValue"
            ],
            "Resource": "arn:aws:secretsmanager:us-west-2:109561199089:secret:*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "sts:AssumeRole"
            ],
            "Resource": "arn:aws:iam::*:role/grafana-cloudwatch-read-only"
        }
    ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "grafana_server_policy" {
  role       = aws_iam_role.ecsTaskExecutionRole.name
  policy_arn = aws_iam_policy.grafana_server_policy.arn
}

resource "aws_iam_role_policy_attachment" "ecsTaskExecutionRole_policy" {
  role       = aws_iam_role.ecsTaskExecutionRole.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_cloudwatch_log_group" "logs" {
  name              = "/fargate/service/grafana-server-${var.environment}"
  retention_in_days = "14"
}
