resource "aws_appautoscaling_target" "grafana_image_renderer" {
  service_namespace  = "ecs"
  resource_id        = "service/${aws_ecs_cluster.grafana_image_renderer.name}/${aws_ecs_service.grafana_image_renderer.name}"
  scalable_dimension = "ecs:service:DesiredCount"
  max_capacity       = "12"
  min_capacity       = "6"
}

resource "aws_ecs_cluster" "grafana_image_renderer" {
  name = "grafana-image-renderer"
}

resource "aws_ecs_task_definition" "grafana_image_renderer" {
  family                   = "grafana-image-renderer-${var.environment}"
  network_mode             = "awsvpc"
  requires_compatibilities = ["FARGATE"]
  cpu                      = "4096"
  memory                   = "16384"
  execution_role_arn       = aws_iam_role.grafana_image_renderer.arn
  task_role_arn            = aws_iam_role.grafana_image_renderer.arn

  container_definitions = <<DEFINITION
[
  {
    "name": "grafana-image-renderer-${var.environment}",
    "cpu": 4096,
    "image": "${var.image_renderer_image}",
    "memory": 16384,
    "name": "grafana-image-renderer-${var.environment}",
    "essential": true,
    "networkMode": "awsvpc",
    "portMappings": [
      {
        "containerPort": 8081
      }
    ],
    "environment": [
      {
        "name": "ENVIRONMENT",
        "value": "${var.environment}"
      },
      {
        "name": "AWS_STS_REGIONAL_ENDPOINTS",
        "value": "regional"
      }
    ],
    "logConfiguration": {
      "logDriver": "awslogs",
      "options": {
        "awslogs-group": "/fargate/service/grafana-image-renderer-${var.environment}",
        "awslogs-region": "us-west-2",
        "awslogs-stream-prefix": "ecs"
      }
    }
  }
]
DEFINITION
}

resource "aws_ecs_service" "grafana_image_renderer" {
  name             = "grafana-image-renderer-${var.environment}"
  cluster          = aws_ecs_cluster.grafana_image_renderer.id
  task_definition  = aws_ecs_task_definition.grafana_image_renderer.arn
  desired_count    = var.service_min_instances
  launch_type      = "FARGATE"
  platform_version = "1.4.0"

  network_configuration {
    security_groups = var.vpc_security_group_ids
    subnets         = var.subnet_ids
  }

  load_balancer {
    target_group_arn = aws_alb_target_group.grafana_image_renderer.arn
    container_name   = "grafana-image-renderer-${var.environment}"
    container_port   = "8081"
  }

  depends_on = [
    aws_alb_listener.grafana_image_renderer,
  ]

  # In most cases we won't want our terraform run to modify the task definition after the initial run
  # Things like the image will be modified by deploys etc potentially, and we dont want to run terraform every time
  lifecycle {
    ignore_changes = [task_definition, desired_count]
  }
}

# For task executions
resource "aws_iam_role" "grafana_image_renderer" {
  name = "grafana-image-renderer-${var.environment}"
  path = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": [
          "ec2.amazonaws.com",
          "ecs-tasks.amazonaws.com",
          "events.amazonaws.com"
        ]
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "grafana_image_renderer_task_policy" {
  role       = aws_iam_role.grafana_image_renderer.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonECSTaskExecutionRolePolicy"
}

resource "aws_cloudwatch_log_group" "grafana_image_renderer_logs" {
  name              = "/fargate/service/grafana-image-renderer-${var.environment}"
  retention_in_days = "7"
}
