### lb

resource "aws_alb" "grafana_server" {
  name            = "ecs-grafana-server"
  internal        = true
  security_groups = var.vpc_security_group_ids
  subnets         = var.subnet_ids
}

resource "aws_alb_target_group" "grafana_server" {
  name        = "grafana-server"
  port        = 3000
  protocol    = "HTTP"
  vpc_id      = var.vpc_id
  target_type = "ip"

  stickiness {
    cookie_duration = 1800
    cookie_name     = "grafana_session"
    enabled         = true
    type            = "app_cookie"
  }

  health_check {
    healthy_threshold   = 5
    unhealthy_threshold = 3
    timeout             = 5
    interval            = 10
    path                = "/logout"
    matcher             = "200,302"
  }
}

# Redirect all traffic from the lb to the target group
resource "aws_alb_listener" "grafana_server" {
  load_balancer_arn = aws_alb.grafana_server.arn
  port              = "443"
  protocol          = "HTTPS"
  certificate_arn   = "arn:aws:acm:us-west-2:109561199089:certificate/29565916-cad2-4b60-9dd8-e5c84f29c2b1"

  default_action {
    target_group_arn = aws_alb_target_group.grafana_server.id
    type             = "forward"
  }
}

# Redirect all traffic from port 80 to 443.
resource "aws_alb_listener" "grafana_server_redirect" {
  load_balancer_arn = aws_alb.grafana_server.arn
  port              = "80"
  protocol          = "HTTP"

  default_action {
    type = "redirect"

    redirect {
      port        = "443"
      protocol    = "HTTPS"
      status_code = "HTTP_301"
    }
  }
}


# DNS for ALB - allows image-renderer to call back in, locally.
resource "aws_route53_record" "lb_dns" {
  zone_id = "Z03975331D5BVQ9TB97AT" # grafana.xarth.tv. (private)
  name    = ""                      # grafana.xarth.tv
  type    = "A"

  alias {
    name                   = aws_alb.grafana_server.dns_name
    zone_id                = aws_alb.grafana_server.zone_id
    evaluate_target_health = false
  }
}

# We can use the privatelink to give other AWS accounts (and VPN) access to Grafana.
module "privatelink" {
  source      = "git::git+ssh://git@git.xarth.tv/terraform-modules/privatelink-alb.git?ref=v0.0.6"
  account     = "twitch-grafana-prod"
  service     = "ecs-grafana"
  environment = var.environment
  dns_name    = aws_alb.grafana_server.dns_name
  vpc         = var.vpc_id
  subnets     = var.subnet_ids
}

# output
output "alb_name" {
  value = aws_alb.grafana_server.dns_name
}

output "vpc_endpoint_name" {
  value = module.privatelink.vpc_endpoint_name
}
