terraform {
  backend "s3" {
    bucket  = "twitch-grafana-prod"
    key     = "tfstate/systems/systems-terraform/production/grafana-prod/ldap-sync/terraform.tfstate"
    region  = "us-west-2"
    profile = "twitch-grafana-prod"
  }
}

provider "aws" {
  profile = var.account
  alias   = "twitch-account-name"
  region  = var.region
}

data "aws_ssm_parameter" "grafana_api_key" {
  provider = aws.twitch-account-name
  name     = "/${var.account}/grafana-api-key.txt"
}

resource "aws_lambda_function" "ldap_sync" {
  provider                       = aws.twitch-account-name
  function_name                  = "${var.service}-${var.environment}"
  s3_bucket                      = var.account
  s3_key                         = "lambda/${var.service}.zip"
  description                    = "Keep Grafana Teams in Sync with LDAP Groups"
  handler                        = var.service
  role                           = aws_iam_role.ldap_sync.arn
  runtime                        = "go1.x"
  timeout                        = "60"
  memory_size                    = "192"
  reserved_concurrent_executions = 1

  vpc_config {
    subnet_ids         = var.subnet_ids
    security_group_ids = var.vpc_security_group_ids
  }

  environment {
    variables = {
      GRAFANA_URL     = "https://osiris.grafana.xarth.tv"
      LAMBDA          = "true"
      GRAFANA_API_KEY = data.aws_ssm_parameter.grafana_api_key.value
    }
  }

  tags = {
    s3_package_path = "https://${var.account}.s3-${var.region}.amazonaws.com/lambda/${var.service}.zip"
  }
}

resource "aws_iam_role" "ldap_sync" {
  provider = aws.twitch-account-name
  name     = "${var.service}-${var.environment}-Lambda"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Principal": {
      "Service": "lambda.amazonaws.com"
    },
    "Action": "sts:AssumeRole"
  }
}
EOF
}

resource "aws_iam_role_policy_attachment" "ldap_sync_vpc_access" {
  provider   = aws.twitch-account-name
  role       = aws_iam_role.ldap_sync.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy_attachment" "ldap_sync_basic_exec" {
  provider   = aws.twitch-account-name
  role       = aws_iam_role.ldap_sync.name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
}

# Jenkins uses this role to update the lambda code and redeploy (upgrade) the lambda.
resource "aws_iam_role" "ldap_sync_jenkins" {
  provider = aws.twitch-account-name
  name     = "${var.service}-${var.environment}-Jenkins"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Principal": {
      "AWS": "${var.jenkins_arn}"
    },
    "Action": "sts:AssumeRole"
  }
}
EOF
}

resource "aws_iam_role_policy_attachment" "jenkins_ldap_sync_attach" {
  provider   = aws.twitch-account-name
  role       = aws_iam_role.ldap_sync_jenkins.name
  policy_arn = aws_iam_policy.jenkins_ldap_sync.arn
}

data "aws_caller_identity" "current" {
  provider = aws.twitch-account-name
}

resource "aws_iam_policy" "jenkins_ldap_sync" {
  provider    = aws.twitch-account-name
  name        = "${var.service}-${var.environment}-Jenkins"
  description = "Allow jenkins to reload the ldap sync lambda."
  policy      = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Sid": "JenkinsLambdaFunctionUpdates",
      "Effect": "Allow",
      "Action": [
        "lambda:UpdateFunctionCode",
        "lambda:UpdateFunctionConfiguration"
      ],
      "Resource": "arn:aws:lambda:${var.region}:${data.aws_caller_identity.current.account_id}:function:${aws_lambda_function.ldap_sync.function_name}"
    },
    {
      "Sid": "JenkinsS3FunctionUpdates",
      "Effect": "Allow",
      "Action": [
        "s3:PutObject",
        "s3:GetObject"
      ],
      "Resource": "arn:aws:s3:::${var.account}/lambda/*"
    }
  ]
}
EOF
}

/* The following resources create a trigger that invokes the lambda when a user logs into Grafana. */

resource "aws_lambda_permission" "ldap_sync_trigger" {
  provider      = aws.twitch-account-name
  statement_id  = "${var.service}-allow-cloudwatch"
  action        = "lambda:InvokeFunction"
  function_name = aws_lambda_function.ldap_sync.arn
  principal     = "logs.${var.region}.amazonaws.com"
  source_arn    = "arn:aws:logs:${var.region}:109561199089:log-group:/fargate/service/grafana-server-production*"
}

resource "aws_cloudwatch_log_subscription_filter" "ldap_sync_trigger" {
  provider        = aws.twitch-account-name
  depends_on      = [aws_lambda_permission.ldap_sync_trigger]
  name            = "${var.service}-${var.environment}-user-login"
  log_group_name  = "/fargate/service/grafana-server-production"
  filter_pattern  = "\"Successful Login\""
  destination_arn = aws_lambda_function.ldap_sync.arn
}
