'''
Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.

Licensed under the Apache License, Version 2.0 (the "License").
You may not use this file except in compliance with the License.
A copy of the License is located at

    http://aws.amazon.com/apache2.0/

or in the "license" file accompanying this file.
This file is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
either express or implied. See the License for
the specific language governing permissions
and limitations under the License.
'''

# Support module for deploying and destroy aurora databases.

import re

_SUPPORTED_ENGINES = ['aurora', 'aurora-mysql', 'aurora-postgresql']


class SnapshotToolException(Exception):
    pass


def filter_clusters(pattern, cluster_list):
    # Takes the response from describe-db-clusters and filters according to pattern in DBClusterIdentifier
    filtered_list = []

    for cluster in cluster_list['DBClusters']:

        if pattern == 'ALL_CLUSTERS' and cluster['Engine'] in _SUPPORTED_ENGINES:
            filtered_list.append(cluster)

        else:
            match = re.search(pattern, cluster['DBClusterIdentifier'])

            if match and cluster['Engine'] in _SUPPORTED_ENGINES:
                filtered_list.append(cluster)

    return filtered_list


def get_own_snapshots_dest(pattern, response):
    """ Returns a dict  with local snapshots, filtered by pattern,
    with DBClusterSnapshotIdentifier as key and Arn, Status as attributes """
    filtered = {}
    for snapshot in response['DBClusterSnapshots']:

        if (snapshot['SnapshotType'] == 'manual' and
            re.search(pattern, snapshot['DBClusterIdentifier']) and
           snapshot['Engine'] in _SUPPORTED_ENGINES):
            filtered[snapshot['DBClusterSnapshotIdentifier']] = {
                'Arn': snapshot['DBClusterSnapshotArn'],
                'Status': snapshot['Status'],
                'StorageEncrypted': snapshot['StorageEncrypted'],
                'DBClusterIdentifier': snapshot['DBClusterIdentifier']}

            if snapshot['StorageEncrypted'] is True:
                filtered[snapshot['DBClusterSnapshotIdentifier']
                         ]['KmsKeyId'] = snapshot['KmsKeyId']

        elif (snapshot['SnapshotType'] == 'manual' and
              pattern == 'ALL_SNAPSHOTS' and
              snapshot['Engine'] in _SUPPORTED_ENGINES):
            filtered[snapshot['DBClusterSnapshotIdentifier']] = {
                'Arn': snapshot['DBClusterSnapshotArn'],
                'Status': snapshot['Status'],
                'StorageEncrypted': snapshot['StorageEncrypted'],
                'DBClusterIdentifier': snapshot['DBClusterIdentifier']}

            if snapshot['StorageEncrypted'] is True:
                filtered[snapshot['DBClusterSnapshotIdentifier']
                         ]['KmsKeyId'] = snapshot['KmsKeyId']

    return filtered


def paginate_api_call(client, api_call, objecttype, *args, **kwargs):
    """ Takes an RDS boto client and paginates through api_call calls
    and returns a list of objects of objecttype. """
    response = {}
    response[objecttype] = []

    # Create a paginator
    paginator = client.get_paginator(api_call)

    # Create a PageIterator from the Paginator
    page_iterator = paginator.paginate(**kwargs)
    for page in page_iterator:
        for item in page[objecttype]:
            response[objecttype].append(item)

    return response
