#!/usr/bin/env bash

set -euo pipefail

cd "$(dirname "$0")"

# if the flock utility is available, avoid running concurrently
if type -t flock >/dev/null; then
  # acquire fd to arbitrary lock file (doesn't need a lock)
  exec {lockfd}>/tmp/iprep-scores-import.lock
  # now get a lock for the fd
  if ! flock --nonblock "$lockfd"; then
    echo "another process is already running, aborting"
    exit 1
  fi
  # when the current process exits, the fd will be closed
  # and the lock will be released automatically
fi

# source config files, if any
if compgen -G '/etc/iprep-scores-import/*.conf' >/dev/null; then
  for f in /etc/iprep-scores-import/*.conf; do
    . "$f"
  done
fi

IPREP_SCORES_DIR="${IPREP_SCORES_DIR:-s3://viewcounts-rep-scores/scores-iprep}"
IPREP_SCORES_OVERRIDE_DBOPT="${IPREP_SCORES_OVERRIDE_DBOPT:-gvc.iprep.update.force_url}"
IPREP_SCORES_PATH="${IPREP_SCORES_PATH:-$IPREP_SCORES_DIR/latest}"
ASNREP_SCORES_PREFIX="${ASNREP_SCORES_PREFIX:-s3://backendabuse/scores-update/scores-}"

# note: user nobody needs to be able to write to the dir
#       because sqlite creates temporary journal files.
[ -v ASNREP_DB_PATH ] || ASNREP_DB_PATH=/var/lib/iprep-scores-import/vba_score/vba_score.db

# user nobody needs to be able to write to the dir because we add files to it.
[ -v IPREP_DB_DIR ] || IPREP_DB_DIR=/var/lib/iprep-scores-import/vbi_score

RUN_ON_REP_UPDATE="${RUN_ON_REP_UPDATE:-}"
RUN_ON_REP_UPDATE_DELAY="${RUN_ON_REP_UPDATE_DELAY:-0}"

tee-stderr-on-failure() {
  # we're a cronjob: our stdout goes into a logfile, our stderr gets mailed.
  # we want everything to be logged, but only get mail on bad exit statuses.

  local output
  # child's stdout is captured, child's stderr goes to our stdout,
  # they're both the same text
  { output="$( set -o pipefail ; "$@" 2>&1 | tee >(cat >&2) )" ; } 2>&1
  if [[ $? -ne 0 ]]; then
    # on error, repeat output to our stderr
    printf >&2 "%s\n" "$output"
    return 1
  fi
}

updated=()

get-mtime() {
  stat --format=%y "$1" 2>/dev/null || echo "absent"
}

run-update-script() {
  local target="$1"
  local cmd=("${@:2}")
  local mtime_before
  mtime_before="$(get-mtime "$target")"
  S3_USE_SIGV4=1 tee-stderr-on-failure setuidgid nobody "./virtualenv/bin/python" "${cmd[@]}" || return 1
  local mtime_after="$(get-mtime "$target")"
  if [[ "$mtime_before" != "$mtime_after" ]]; then
    updated+=("$target (previously $mtime_before);")
  fi
}

restart-services() {
  if [[ -z "$RUN_ON_REP_UPDATE" ]]; then
    return
  fi
  if [[ "${#updated[@]}" -eq 0 ]]; then
    echo "no changes; not running restart commands"
    return
  fi

  echo "changed:"
  for f in "${updated[@]}"; do
    echo "    $f"
  done
  local status=0
  if [[ "$RUN_ON_REP_UPDATE_DELAY" -ne 0 ]]; then
    echo "waiting $RUN_ON_REP_UPDATE_DELAY seconds before running scripts..."
    sleep "$RUN_ON_REP_UPDATE_DELAY"
  fi
  for cmd in $RUN_ON_REP_UPDATE; do
    if [[ -x "$cmd" ]]; then
      echo "running $cmd"
      "$cmd" || status=1
    else
      echo "not found, skipping: $cmd";
      status=1
    fi
  done
  echo "done"
  return "$status"
}

export AWS_ACCESS_KEY_ID AWS_SECRET_ACCESS_KEY

[[ -v IPREP_AWS_ACCESS_KEY_ID ]] && AWS_ACCESS_KEY_ID=$IPREP_AWS_ACCESS_KEY_ID
[[ -v IPREP_AWS_SECRET_ACCESS_KEY ]] && AWS_SECRET_ACCESS_KEY=$IPREP_AWS_SECRET_ACCESS_KEY

status=0
run-update-script "$IPREP_DB_DIR/vbi_score.db" vbi_score.py \
  --input-s3="$IPREP_SCORES_PATH" \
  --input-s3-override-dbopt="$IPREP_SCORES_OVERRIDE_DBOPT" \
  --destination-dir="$IPREP_DB_DIR" || status=1

run-update-script "$ASNREP_DB_PATH" vba_score.py \
  --input-s3="$ASNREP_SCORES_PREFIX" \
  --database-url=sqlite:///"$ASNREP_DB_PATH" || status=1

if [[ $status -eq 0 ]]; then
  tee-stderr-on-failure restart-services || status=1
fi

exit $status
