package envoy

import "encoding/json"

// A Condition maps a State to a level of healthiness.
type Condition int16

// The constants for condition levels below are set in order of severity
// c.f. Condition.IsWorse
// Smaller number = more severe

const (
	ConditionCritical Condition = iota
	ConditionUnknown
	ConditionWarning
	ConditionNormal
)

var conditionToName = map[Condition]string{
	ConditionNormal:   "NORMAL",
	ConditionWarning:  "WARNING",
	ConditionCritical: "CRITICAL",
	ConditionUnknown:  "UNKNOWN",
}

// String implements the String interface for Condition
func (c Condition) String() string {
	return conditionToName[c]
}

// IsWorse returns true if the condition c is worse then the
// condition other. The ordering is defined as
//
// Normal < Warning < Unknown < Critical
//
// Critical is the worst condition
//
// Unknown could be critical, but we aren't sure
func (c Condition) IsWorse(other Condition) bool {
	return c < other
}

// MarshalJSON implements the JSON interface for Condition
func (c Condition) MarshalJSON() ([]byte, error) {
	return json.Marshal(c.String())
}
