package envoy

import (
	"encoding/json"
	"log"
	"net"
	"net/http"
	"time"
)

const (
	restartTimeout time.Duration = 5 * time.Second
	healthPattern  string        = "/health"
)

// HealthHandler is the HTTP handler used by envoy to respond
// to queries on the path /health. It responds with text/plain
// and application/json Content-Types.
func (e *Envoy) HealthHandler(w http.ResponseWriter, r *http.Request) {
	samples := e.newStateSnapshots()
	globalCond := samples.GlobalSnapshot.Current
	if globalCond.IsWorse(ConditionWarning) {
		w.WriteHeader(http.StatusServiceUnavailable)
	}

	if r.Header.Get("Accept") == "application/json" || r.Header.Get("Content-Type") == "application/json" {
		w.Header().Set("Content-Type", "application/json")
		enc := json.NewEncoder(w)
		enc.Encode(samples)
	} else {
		w.Header().Set("Content-Type", "text/plain")
		w.Write([]byte(samples.String()))
	}
}

func (e *Envoy) runServer(conf *HTTPServerConf) {
	http.HandleFunc(healthPattern, e.HealthHandler)

	for {
		err := http.ListenAndServe(net.JoinHostPort(conf.Host, conf.Port), nil)
		log.Printf("ENVOY ERROR: %+v\n", err)
		time.Sleep(restartTimeout)
	}
}
