package envoy

import (
	"fmt"
	"sync"
	"time"
)

const ChangeBufferLen = 8

// A State represents one component of your service's health.
// Each state should reflect the condition of one specific dependency
// or metric of the service.
// Envoy will "bubble up" any changes in condition of any state such
// that the global condition reflects the worst condition of all the
// states.
type State struct {
	name        string                  // The name or key of the state
	description string                  // The description of the state
	latest      Condition               // The last condition we received
	messages    map[Condition]string    // The last message set from an event update
	timings     map[Condition]time.Time // List of most recent conditions and timings
	parent      *Envoy
	sync.RWMutex
}

func (s *State) update(change *StateChange) {
	s.RWMutex.Lock()
	defer s.RWMutex.Unlock()

	transition := change.Condition != s.latest

	s.latest = change.Condition
	s.messages[change.Condition] = change.Message
	s.timings[change.Condition] = change.changedAt

	if transition {
		select {
		case s.parent.transitions <- struct{}{}:
		default:
		}
	}
}

func (s *State) GoString() string {
	s.RWMutex.RLock()
	defer s.RWMutex.RUnlock()

	return fmt.Sprintf(`&envoy.State{name:%q, description:%q, latest:%v, messages:%#v, timings:%#v, parent:(%T)(%p), RWMutex:%#v`, s.name, s.description, s.latest, s.messages, s.timings, s.parent, s.parent, s.RWMutex)
}

func (s *State) String() string {
	snapshot := s.Snapshot()

	s.RWMutex.RLock()
	defer s.RWMutex.RUnlock()

	return fmt.Sprintf("State[%v]{description:%q, messages:%d, condition:%v}", s.name, s.description, len(s.messages), snapshot.Current)
}

func (s *State) Snapshot() StateSnapshot {
	s.RWMutex.RLock()
	defer s.RWMutex.RUnlock()

	worst := s.latest
	history := make(history)

	// scan all conditions and try to find which we should be in
	// we want the most severe (worst) condition that has happened in the last windowLength
	// but if none have occured in that timespan, then we just take the most recent
	for c, t := range s.timings {
		history[c] = t

		if time.Since(t) > s.parent.windowLength {
			continue
		}

		if c.IsWorse(worst) {
			worst = c
		}
	}

	return StateSnapshot{
		Name:        s.name,
		Description: s.description,
		Message:     s.messages[worst],
		Current:     worst,
		History:     history,
	}
}

func (s *State) set(condition Condition, message string) {
	updates := make(StateUpdates)
	updates[s] = &StateChange{
		Condition: condition,
		Message:   message,
	}
	s.parent.Update(updates)
}

// Set the condition of the state to Normal with an explanation
func (s *State) Normal(message string) {
	s.set(ConditionNormal, message)
}

// Set the condition of the state to Warning with an explanation
func (s *State) Warning(message string) {
	s.set(ConditionWarning, message)
}

// Set the condition of the state to Critical with an explanation
func (s *State) Critical(message string) {
	s.set(ConditionCritical, message)
}

// Set the condition of the state to Unknown with an explanation
func (s *State) Unknown(message string) {
	s.set(ConditionUnknown, message)
}

// NewState creates a new envoy state. States should have a machine-consumable
// name and a human-readable description. The state will be immediately added to
// the query output for envoy with the condition startCon.
//
// The starting condition should reflect what the application knows when the state
// is created. If, for example, the state represents the existence of some backends
// that the service relies on, but the service has not yet tried to query those
// backends, then it should use the condition ConditionUnknown.
func (e *Envoy) NewState(name, description string, startCon Condition) *State {
	e.Lock()
	defer e.Unlock()

	messages := make(map[Condition]string)
	timings := make(map[Condition]time.Time)

	messages[startCon] = "Starting"
	timings[startCon] = time.Now()

	s := &State{
		name:        name,
		description: description,
		latest:      startCon,
		messages:    messages,
		timings:     timings,
		parent:      e,
	}

	e.states = append(e.states, s)

	return s
}
