package api

import (
	"encoding/json"
	"log"
	"net/http"
	"regexp"

	"code.justin.tv/video/streamlog/pkg/statsd"

	"code.justin.tv/abuse/shodan/iprep"
)

type iprepHandler struct {
	pathRegexp *regexp.Regexp
	repSource  iprep.Source
}

type fracViewsResult struct {
	FracViews *float64 `json:"frac_views,omitempty"`
	Error     string   `json:"error,omitempty"`
}

func (h *iprepHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	statsd.Inc("api.request.total", 1, 1)
	matches := h.pathRegexp.FindStringSubmatch(r.URL.Path)
	if matches == nil {
		w.WriteHeader(400)
		w.Write([]byte("{\"error\":\"couldn't parse ip address\"}"))
		return
	}
	ip := matches[1]

	fracViews, err := h.repSource.FracViewsForIP(ip, nil)
	if fracViews < 0 {
		fracViews = 0
	}
	if fracViews > 1 {
		fracViews = 1
	}
	var result fracViewsResult
	if err != nil {
		result.Error = err.Error()
	} else {
		result.FracViews = &fracViews
	}
	statsd.Inc("api.response.total", 1, 1)
	bytes, err := json.Marshal(result)
	if err != nil {
		log.Printf("couldn't serialize result (%v): %v\n", result, err)
		w.WriteHeader(500)
		return
	}

	if result.FracViews != nil {
		statsd.Inc("api.response.ok", 1, 1)
	} else {
		statsd.Inc("api.response.err", 1, 1)
		w.WriteHeader(500)
	}

	w.Write([]byte(bytes))
}

func NewHandler(pathPrefix string, repSource iprep.Source) http.Handler {
	const pathSuffix = "/FracViews"
	pathRegexp := regexp.MustCompile(
		"^" + regexp.QuoteMeta(pathPrefix) +
			"(\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3})" +
			regexp.QuoteMeta(pathSuffix) + "$")
	return &iprepHandler{
		pathRegexp: pathRegexp,
		repSource:  repSource,
	}
}
