// Takes forever to tun in our build system with -race
// I ran it locally though and there are no race conditions here. Yay.

// +build !race

package dbopt

import (
	"encoding/json"
	"log"
	"net/http"
	"net/url"
	"reflect"
	"strconv"
	"sync/atomic"
	"testing"
	"time"
)

const (
	testServer = "http://localhost:5555/dboption/all/"
)

var (
	requestHandler func(http.ResponseWriter, *http.Request)
)

func TestMain(m *testing.M) {
	u, err := url.Parse(testServer)
	if err != nil {
		log.Fatalf("Failed to parse testServer URL %s: %s", testServer, err)
	}

	mux := http.NewServeMux()
	log.Printf(u.Path)
	mux.HandleFunc("/", func(w http.ResponseWriter, r *http.Request) {
		log.Printf("%s", r.URL)
		requestHandler(w, r)
	})
	go http.ListenAndServe(":5555", mux)
	//os.Exit(m.Run())
}

func makeClient(t *testing.T, freq time.Duration, patterns ...string) *Client {
	url := testServer
	client := NewClient(freq, &url, patterns...)
	errC := make(chan error)

	go errorHandler(t, client, errC)
	client.ErrorHandler(errC)
	return client
}

func errorHandler(t *testing.T, client *Client, errC chan error) {
	for {
		err := <-errC

		if err != nil {
			log.Panicf("Error from client: %#v", err)

			for c := range client.listeners {
				close(c)
			}
		}
	}
}

func writeJSONResponse(w http.ResponseWriter, v interface{}) {
	out, err := json.Marshal(v)
	if err != nil {
		w.WriteHeader(400)
		w.Write([]byte(err.Error()))
		return
	}

	w.Header().Add("Content-Type", "application/json")
	w.Header().Add("Content-Length", strconv.Itoa(len(out)))

	w.Write(out)
}

func reqHandlerForData(aVal *atomic.Value) func(w http.ResponseWriter, r *http.Request) {
	return func(w http.ResponseWriter, r *http.Request) {
		values := aVal.Load().(map[string]interface{})
		obj := []map[string]interface{}{values}
		writeJSONResponse(w, obj)
	}
}

func assertEqual(t *testing.T, expected interface{}, actual interface{}, e string) {
	if !reflect.DeepEqual(expected, actual) {
		t.Fatalf("Mismatch for %s: expected %#v, but got %#v", e, expected, actual)
	}
}

func assertEqualAVal(t *testing.T, expected interface{}, actualAVal *atomic.Value, e string) {
	assertEqual(t, expected, actualAVal.Load().(map[string]interface{}), e)
}

func copyMap(m map[string]interface{}) map[string]interface{} {
	result := make(map[string]interface{}, len(m))
	for k, v := range m {
		result[k] = v
	}
	return result
}

func updateHandler() (chan Update, *atomic.Value) {
	var result atomic.Value
	result.Store(make(map[string]interface{}))
	c := make(chan Update)

	go func() {
		newResult := copyMap(result.Load().(map[string]interface{}))

		for update := range c {
			newResult[update.Name] = update.Value
			result.Store(copyMap(newResult))
		}
	}()

	return c, &result
}

func updateValues(values *atomic.Value, updates map[string]interface{}) {
	newValues := copyMap(values.Load().(map[string]interface{}))
	for k, v := range updates {
		newValues[k] = v
	}
	values.Store(newValues)
}

func TestUpdates(t *testing.T) {
	var values atomic.Value
	values.Store(map[string]interface{}{
		"foo.bar1": "flgr",
		"foo.bar2": 1.0,
		"foo.qux1": -3.4,
		"foo.qux2": nil,
		"f":        "missing",
	})

	requestHandler = reqHandlerForData(&values)
	freq := 10 * time.Millisecond

	client := makeClient(t, freq, "foo.*")
	c1, m1 := updateHandler()
	c2, m2 := updateHandler()
	c3, m3 := updateHandler()
	client.RegisterChannel(c1, "foo.bar*")
	client.RegisterChannel(c2, "foo.qux*")

	time.Sleep(freq * 2)
	t.Logf("%#v", m1.Load())

	assertEqualAVal(t, map[string]interface{}{
		"foo.bar1": "flgr",
		"foo.bar2": 1.0,
	}, m1, "foo.bar* values")

	assertEqualAVal(t, map[string]interface{}{
		"foo.qux1": -3.4,
		"foo.qux2": nil,
	}, m2, "foo.qux* values")

	updateValues(&values, map[string]interface{}{
		"foo.barbar": "cohan",
		"foo.bar2":   42.0,
		"foo.qux1":   0.0,
		"foo.qux2":   0.0,
		"foo.qux":    -1.0,
	})

	client.RegisterChannel(c3, "f*")
	time.Sleep(freq * 2)

	assertEqualAVal(t, map[string]interface{}{
		"foo.bar1":   "flgr",
		"foo.bar2":   42.0,
		"foo.barbar": "cohan",
	}, m1, "updated foo.bar* values")

	assertEqualAVal(t, map[string]interface{}{
		"foo.qux1": 0.0,
		"foo.qux2": 0.0,
		"foo.qux":  -1.0,
	}, m2, "updated foo.qux* values")

	assertEqualAVal(t, values.Load(), m3, "all values")
}
