package dbopt

import (
	"encoding/json"
	"log"
	"sync"
)

type Map struct {
	values map[string]interface{}
	sync.RWMutex
}

func DefaultMap(patterns ...string) *Map {
	return MapFor(DefaultClient, patterns...)
}

func MapFor(cl *Client, patterns ...string) *Map {
	r := Map{values: make(map[string]interface{})}
	c := make(chan Update, 100)
	cl.RegisterChannel(c, patterns...)

	go func() {
		for u := range c {
			r.apply(u)
		}
	}()

	return &r
}

func (s *Map) apply(u Update) {
	s.Lock()
	defer s.Unlock()
	s.values[u.Name] = u.Value
}

func (s *Map) Get(name string, def interface{}) interface{} {
	s.RLock()
	defer s.RUnlock()
	if val, ok := s.values[name]; ok {
		return val
	}
	return def
}

func doErr(format string, args ...interface{}) {
	log.Printf(format, args...)
}

func (s *Map) GetString(name string, def string) string {
	switch t := s.Get(name, def).(type) {
	case string:
		return t
	case json.Number:
		return t.String()
	default:
		doErr("Failed to parse %#v as string", t)
		return def
	}
}

func (s *Map) GetFloat64(name string, def float64) float64 {
	switch t := s.Get(name, nil).(type) {
	case json.Number:
		val, err := t.Float64()
		if err == nil {
			return val
		}
		doErr("Failed to parse %#v as float64: %s", t, err)
	case nil:
		/* OK */
	default:
		doErr("Failed to parse %#v as float64", t)
	}
	return def
}

func (s *Map) GetInt64(name string, def int64) int64 {
	switch t := s.Get(name, nil).(type) {
	case json.Number:
		// Unfortunately, we have to parse ints as floats,
		// because Usher sometimes messes them up :(
		val, err := t.Float64()
		if err == nil {
			return int64(val)
		}
		doErr("Failed to parse %#v as int64: %s", t, err)
	case nil:
		/* OK */
	default:
		doErr("Failed to parse %#v as int64", t)
	}
	return def
}

// TODO: Should use https://golang.org/pkg/encoding/json/#Decoder.UseNumber
// for all this.
func (s *Map) GetFloat32(name string, def float32) float32 {
	return float32(s.GetFloat64(name, float64(def)))
}

func (s *Map) GetInt(name string, def int) int {
	return int(s.GetInt64(name, int64(def)))
}

func (s *Map) GetInt32(name string, def int32) int32 {
	return int32(s.GetInt64(name, int64(def)))
}
