module Bits
  class ProductsController < Bits::BaseController
    before_action :set_product, only: [:edit]
    before_action :set_enums, only: [:new, :create, :edit]

    def index
      authorize Product

      begin
        @products = Product.all
      rescue StandardError
        @products = []
        flash[:error] = "Error loading Bits products"
      end
    end

    def new
      @product = Product.new
      authorize @product

      begin
        @bits_types = Bits::Type.all
      rescue StandardError
        @bits_types = []
        flash[:error] = "Error loading existing Bits types"
      end
    end

    def create
      @product = build_product product_params
      authorize @product

      begin
        @bits_types = Bits::Type.all
      rescue StandardError
        @bits_types = []
        flash[:error] = "Error loading existing Bits types"
      end

      if @product.save
        flash[:success] = "Success created new Bits Product [#{@product.display_name}]"
        redirect_to bits_products_path
      else
        flash.now[:danger] = @product.errors.full_messages.to_sentence
        render :new
      end
    end

    def edit
      @bits_types = Bits::Type.all
    end

    def update
      @product = build_product product_params
      authorize @product

      if @product.save
        flash[:success] = "Success updated Bits Product [#{@product.display_name}]"
        redirect_to bits_products_path
      else
        flash.now[:danger] = @product.errors.full_messages.to_sentence
        render :edit
      end
    end

    protected

    def build_product(params)
      product = Product.new(
        id: params[:id],
        display_name: params[:display_name],
        platform: params[:platform].to_sym,
        bits_type_id: params[:bits_type_id],
        quantity: params[:quantity].to_i,
        max_purchasable_quantity: params[:max_purchasable_quantity].to_i,
        show_when_logged_out: params[:show_when_logged_out] == "true",
        pricing_id: params[:pricing_id]
      )

      if params[:include_promo] == "true"
        promo = ProductPromo.new(
          id: params[:promo][:id],
          type: params[:promo][:type].to_sym,
          start: Time.find_zone("UTC").strptime(params[:promo][:start], "%Y-%m-%dT%H:%M").utc,
          end: Time.find_zone("UTC").strptime(params[:promo][:end], "%Y-%m-%dT%H:%M").utc
        )
        product.promo = promo
      end

      product
    end

    def set_product
      @product = Product.find(params[:id], params[:platform].to_sym)
      authorize @product
    end

    def set_enums
      @platforms = Product.platforms
      @promo_types = Product.promo_types
    end

    def product_params
      params.require(:bits_product).permit(:product,
                                           :id,
                                           :display_name,
                                           :platform,
                                           :bits_type_id,
                                           :quantity,
                                           :max_purchasable_quantity,
                                           :pricing_id,
                                           :show_when_logged_out,
                                           :include_promo,
                                           promo: [:id, :type, :start, :end])
    end
  end
end
