module Clue
  class ChatPropertiesController < Clue::BaseController
    before_action :set_user, only: [:index, :edit, :update]

    AUDIT_ACTION_UPDATE = "update"
    AUDIT_CHAT_PROPERTIES = "chat_properties"
    AUDIT_LDAP_USER = "ldap_user"

    def index
      authorize ChatProperty
      @chat_properties = if @twitch_user.present?
                           ChatProperty.find(@twitch_user.id)
                         else
                           ChatProperty.new
                         end
    end

    def edit
      authorize ChatProperty
      @chat_properties = ChatProperty.find(params[:id])
    end

    def update
      authorize ChatProperty
      form_params = params[:clue_chat_property]

      new_fields = {
        id: form_params[:id],
        chat_require_verified_account: convert_check_to_bool(form_params[:chat_require_verified_account]),
        hide_chat_links: convert_check_to_bool(form_params[:hide_chat_links]),
        rituals_enabled: convert_check_to_bool(form_params[:rituals_enabled]),
        r9k_only_chat: convert_check_to_bool(form_params[:r9k_only_chat]),
        subscribers_only_chat: convert_check_to_bool(form_params[:subscribers_only_chat]),
        chat_fastsubs: convert_check_to_bool(form_params[:chat_fastsubs]),
      }

      chat_properties = ChatProperty.find(params[:id])

      old_fields = {
        id: chat_properties.id,
        chat_require_verified_account: chat_properties.chat_require_verified_account,
        hide_chat_links: chat_properties.hide_chat_links,
        rituals_enabled: chat_properties.rituals_enabled,
        r9k_only_chat: chat_properties.r9k_only_chat,
        subscribers_only_chat: chat_properties.subscribers_only_chat,
        chat_fastsubs: chat_properties.chat_fastsubs,
      }

      response = chat_properties.update(new_fields)
      if response.success?
        audit_chat_properties(old_fields, new_fields)
        flash[:success] = "Successfully updated chat properties."
      else
        flash[:error] = "Failed to update chat properties."
      end
      redirect_back
    end

    def convert_check_to_bool(check_str)
      if check_str == "1"
        return true
      else
        return false
      end
    end

    protected

    def set_user
      user_login = nil
      user = params[:user].presence
      if user.present?
        if /^\d+$/.match?(user)
          user_id = user
        else
          user_login = user
        end
      end

      if user_id.present?
        @twitch_user = Twitch::User.find(user_id)
      elsif user_login.present?
        @twitch_user = Twitch::User.find_by_login(user_login)
      end
    end

    def audit_chat_properties(prev_chat_properties_params, curr_chat_properties_params)
      changes = []

      curr_chat_properties_params.each do |key, _|
        symbol = key.to_sym
        change = History::ChangeSet.new(
          attribute: symbol,
          old_value: prev_chat_properties_params[symbol],
          new_value: curr_chat_properties_params[symbol]
        )
        changes.push(change)
      end

      audit = History::Audit.new(
        action: AUDIT_ACTION_UPDATE,
        user_type: AUDIT_LDAP_USER,
        user_id: current_user.ldap_login,
        resource_type: AUDIT_CHAT_PROPERTIES,
        resource_id: curr_chat_properties_params[:id],
        changes: changes
      )

      History::AddAudit.add(audit)
    end
  end
end
