module Payments
  class PurchaseRefundsController < Payments::BaseController
    def index
      authorize PurchaseRefund

      return if params[:purchaser_login].blank?

      product_types = []
      if params[:subs]
        product_types << "SUBS"
      end
      if params[:gift_subs]
        product_types << "GIFT_SUBS"
      end
      if params[:bits]
        product_types << "BITS"
      end

      if product_types.empty?
        flash[:error] = "You must choose at least one of Subs, Gift Subs, or Bits to refund"
        return redirect_to(payments_purchase_refunds_path)
      end

      user = Twitch::User.find_by_login(params[:purchaser_login])

      if user.blank?
        flash[:error] = "No user found with username: #{params[:purchaser_login]}"
        return redirect_to(payments_purchase_refunds_path)
      end

      if user.partner_program? || user.developer?
        flash[:error] = 'Mass refunds cannot be performed for partners or extension developers'
        return redirect_to(payments_purchase_refunds_path)
      end

      params[:purchaser_id] = user.id

      response = Payments::RefundPreview.refund_preview(
        params[:purchaser_id],
        current_user.ldap_login,
        current_user.email,
        nil,
        params[:created_at_start],
        Date.strptime(params[:created_at_start], '%Y-%m-%d') + 1.day,
        product_types
      )

      unless valid_refund_preview_response?(response)
        flash[:error] = 'Failed to generate a refund preview. If this issue persists, contact #admin-services on Slack for help'
        return redirect_to(payments_purchase_refunds_path)
      end

      @refund_preview = response.body
      # Gross amount comes back to us in USD and left shifted by two places in base 10
      @refund_preview['gross_amount_usd'] /= 100.0

      track_action! trackable_request
        .assemble_an
        .admin_panel_purchase_mass_refund_search(
          purchaser_login:      params[:purchaser_login],
          purchaser_id:         params[:purchaser_id],
          created_on_utc:       params[:created_at_start],
          refund_reason:        params[:reason],
          include_channel_subs: params[:subs],
          include_gift_subs:    params[:gift_subs],
          include_bits:         params[:bits]
        )
    end

    def create
      authorize PurchaseRefund

      product_types = []
      if params[:subs]
        product_types << "SUBS"
      end
      if params[:gift_subs]
        product_types << "GIFT_SUBS"
      end
      if params[:bits]
        product_types << "BITS"
      end

      success = Payments::RefundAllPayments.refund_all_payments(
        params[:purchaser_id],
        current_user.ldap_login,
        current_user.email,
        nil,
        params[:reason],
        params[:created_at_start],
        Date.strptime(params[:created_at_start], '%Y-%m-%d') + 1.day,
        product_types
      )

      if success
        flash[:success] = "Success! Refund job created."
        track_action! trackable_request
          .assemble_an
          .admin_panel_purchase_mass_refund_create(
            purchaser_id:         params[:purchaser_id],
            created_on_utc:       params[:created_at_start],
            refund_reason:        params[:reason],
            include_channel_subs: params[:subs],
            include_gift_subs:    params[:gift_subs],
            include_bits:         params[:bits]
          )
        redirect_to(payments_purchase_refunds_path)
      else
        flash[:error] = "Failed to create refund job. If this issue persists, please contact #admin-services on Slack"
      end
    end

    private

    def valid_refund_preview_response?(response)
      response.success? && response.body['payments_count'] && response.body['gross_amount_usd']
    end
  end
end
