module Subscriptions
  class EmotesController < Subscriptions::BaseController

    def index
      authorize Emote

      begin
        mako_emotes = Mako::Emote.get_global_emotes
      rescue StandardError
        flash[:error] = "Failed to load global emoticons"
        redirect_back
      end

      @emotes = mako_emotes
      @emote_uploader = Mako::EmoteUploader.new
      @ticket_product = TicketProduct.new
    end

    def deactivate
      authorize Emote

      emote_id = params[:emote_id]

      begin
        emotes = Mako::Emote.get_emoticons_by_emoticon_ids([emote_id])
        emote = emotes[0]

        Mako::Emote.deactivate_emoticon(emote_id)

        track_action! trackable_request
          .assemble_an
          .admin_panel_ticket_product_emote_deactivate(
            emote_id: emote.id,
            code_suffix: emote.suffix,
            group_id: emote.group_id
          )
        flash[:success] = "Successfully deactivated emoticon"
      rescue StandardError
        flash[:error] = "Failed to deactivate emoticon"
      end
      redirect_back
    end

    def upload
      authorize Emote
      form_params = params[:mako_emote_uploader]

      code_prefix = Mako::EmoteUploader::GLOBAL_EMOTE_CODE_PREFIX
      user_id = Mako::EmoteUploader::GLOBAL_EMOTE_OWNER_ID
      unless form_params[:ticket_product_owner_id].blank?
        user_id = form_params[:ticket_product_owner_id]

        # sub emotes use the channel's prefix
        mako_resp = Mako::EmotePrefix.get_prefix(user_id)
        code_prefix = mako_resp.prefix
      end

      group_id = Mako::EmoteUploader::GLOBAL_EMOTE_GROUP_ID
      unless form_params[:group_id].blank?
        group_id = form_params[:group_id]
      end

      options = {
        code_prefix: code_prefix,
        code_suffix: form_params[:code_suffix],
        group_id: group_id,
        data28: form_params[:data_28],
        data56: form_params[:data_56],
        data112: form_params[:data_112],
        user_id: user_id,
        uploader_user_name: current_user.ldap_login,
        uploader_email: current_user.email
      }

      begin
        emote_id = Mako::EmoteUploader.upload(options)

        track_action! trackable_request
          .assemble_an
          .admin_panel_ticket_product_emote_upload(
            user_id: form_params[:user_id],
            code_suffix: form_params[:code_suffix]
          )

        flash[:success] = "Successfully uploaded emote with ID: " + emote_id
      rescue StandardError => e
        flash[:error] = "Failed to upload emote: " + e.message
      end
      redirect_back
    end

    def edit
      authorize Emote

      emote_id = params[:id]

      begin
        emotes = Mako::Emote.get_emoticons_by_emoticon_ids([emote_id])
        if emotes[0].nil?
          raise "no emote found"
        end

        @emote = emotes[0]
      rescue StandardError
        flash[:error] = "Failed to load emote"
        redirect_back
      end
    end

    def update
      authorize Emote
      form_params = params[:mako_emote]
      id = form_params[:id]
      code_suffix = form_params[:suffix]
      group_id = form_params[:group_id]

      begin
        Mako::Emote.update_emote_code_suffix(id, code_suffix)

        track_action! trackable_request
          .assemble_an
          .admin_panel_ticket_product_emote_update(
            emote_id: id,
            code_suffix: code_suffix,
            group_id: group_id
          )
        flash[:success] = "Successfully updated emote"
      rescue StandardError
        flash[:error] = "Failed to update emote"
      end
      redirect_back
    end
  end
end
