module Subscriptions
  class TenuresController < Subscriptions::BaseController
    def index
      authorize Chronobreak::Tenure

      return if params[:tenure].blank?

      form = params[:tenure]
      channel_id = form[:channelID]
      channel_login = form[:channelName]
      user_id = form[:userID]
      user_login = form[:userName]

      channel = if channel_id.present?
                  Twitch::User.find(channel_id)
                elsif channel_login.present?
                  Twitch::User.find_by_login(channel_login)
                end

      user = if user_id.present?
               Twitch::User.find(user_id)
             elsif user_login.present?
               Twitch::User.find_by_login(user_login)
             end

      if channel.blank?
        flash[:error] = "No channel found for: #{channel_id || channel_login}"
        return redirect_to(subscriptions_tenures_path)
      end

      if user.blank?
        flash[:error] = "No user found for: #{user_id || user_login}"
        return redirect_to(subscriptions_tenures_path)
      end

      @channel_id = channel.id.to_s
      @user_id = user.id.to_s

      # Fetch tenure from chronobreak.
      tenure_resp = Chronobreak::Tenure.find_tenure(
        @user_id,
        @channel_id,
        Chronobreak::Tenure::SUB_DOMAIN,
        Chronobreak::Tenure::TENURE_METHODS[:CUMULATIVE]
      )

      # Fetch streak from chronobreak
      streak_resp = Chronobreak::Tenure.find_tenure(
        @user_id,
        @channel_id,
        Chronobreak::Tenure::SUB_DOMAIN,
        Chronobreak::Tenure::TENURE_METHODS[:STREAK]
      )

      # Fetch badge version from subs service
      badge_resp = Substwirp::Badges.get_subscriber_badge({
        user_id: @user_id,
        channel_id: @channel_id
      })

      @tenure = OpenStruct.new(
        tenure: tenure_resp.tenure_value,
        streak: streak_resp.tenure_value,
        badge_type: badge_resp.badge_type,
        badge_version: badge_resp.badge_version,
        anniversary_date: tenure_resp.start,
        channel_id: @channel_id,
        channel_login: channel.login.to_s,
        user_id: @user_id,
        user_login: user.login.to_s,
        days_remaining: tenure_resp.daysRemaining
      )

      # Fetch badge assets from badge service
      @existing_badges = Badge.existing_and_available_for(channel_id: @channel_id)

      @grants = Chronobreak::Tenure.find_grants(@user_id, @channel_id, Chronobreak::Tenure::SUB_DOMAIN)

      @cumulative_tenures = Chronobreak::Tenure.find_tenures(@user_id, @channel_id, Chronobreak::Tenure::SUB_DOMAIN, Chronobreak::Tenure::TENURE_METHODS[:CUMULATIVE])

      @streak_tenures = Chronobreak::Tenure.find_tenures(@user_id, @channel_id, Chronobreak::Tenure::SUB_DOMAIN, Chronobreak::Tenure::TENURE_METHODS[:STREAK])

      @preview_tenure = params[:preview_tenure]

      track_tenure_search!(form: form)
    end

    def delete_grant
      authorize Chronobreak::Tenure

      dcu_id = params[:id].split(":")
      if dcu_id.length != 3
        flash[:error] = "Invalid ID format, need domain:channel:user"
        redirect_back
      end
      origin_id = params[:origin]

      domain = dcu_id[0]
      channel_id = dcu_id[1]
      user_id = dcu_id[2]

      Chronobreak::Tenure.delete_override_grant(
        user_id,
        channel_id,
        origin_id,
        domain
      )

      redirect_back
    end

    def preview_tenure
      authorize Chronobreak::Tenure

      start_date = params[:start]
      end_date = params[:end]
      user_id = params[:user_id]
      channel_id = params[:channel_id]

      if start_date.blank?
        flash[:error] = "Start Date must not be empty"
        redirect_back
      end

      if end_date.blank?
        flash[:error] = "End Date must not be empty"
        redirect_back
      end

      tenure_resp = Chronobreak::Tenure.preview_tenure_with_grant(
        user_id,
        channel_id,
        start_date,
        end_date
      )

      preview_tenure = {
        start_date: start_date,
        end_date: end_date,
        tenure_val: tenure_resp.tenure_value,
        streak_val: tenure_resp.streak_value
      }

      redirect_to action: index, tenure: {channelID: channel_id, userID: user_id}, preview_tenure: preview_tenure
    end

    def upsert_grant
      authorize Chronobreak::Tenure

      start_date = params[:start]
      end_date = params[:end]
      user_id = params[:user_id]
      channel_id = params[:channel_id]

      Chronobreak::Tenure.upsert_override_grant(
        user_id,
        channel_id,
        start_date,
        end_date
      )

      redirect_back
    end

    def track_tenure_search!(form:)
      track_action! trackable_request
        .assemble_an
        .admin_panel_tenures_search(
          user_login:    form[:userName],
          channel_login: form[:channelName],
          user_id:       form[:userID],
          channel_id:    form[:channelID]
        )
    end
  end
end
