module Vods
  class UserPropertyController < ApplicationController
    before_action :set_user
    before_action :set_vod_privacy_property, only: [:update]
    before_action :set_vod_property, only: [:update]

    def edit
    end

    def update
      # Cache the old and new props
      current_properties = @vod_privacy_property.attributes.merge(@vod_property.attributes)
      new_properties = update_user_vod_property_params.merge(update_vod_privacy_property_params)
      # Set the new props on the object
      @vod_privacy_property.attributes = update_vod_privacy_property_params
      @vod_property.attributes = update_user_vod_property_params
      # Try to save the new props
      if @vod_privacy_property.save && @vod_property.save
        # Audit the difference
        audit_user_vod_property_update(current_props: current_properties, new_props: new_properties)
        flash[:success] = "Success"
      else
        errors = @vod_privacy_property.errors.full_messages + @vod_property.errors.full_messages
        flash[:error] = errors.to_sentence
      end
      redirect_to twitch_user_path(params[:id])
    end

    protected

    def update_vod_privacy_property_params
      {
        hide_archives: params[:hide_archives],
        hide_archives_enabled: params[:hide_archives_enabled],
      }
    end

    def update_user_vod_property_params
      {
        save_vods_forever: params[:save_vods_forever],
        skip_upload_moderation: params[:skip_upload_moderation],
        can_upload_unlimited: params[:can_upload_unlimited],
        skip_muting: params[:skip_muting],
      }
    end

    def set_user
      @user = Twitch::User.find params[:id]

      if @user.blank?
        flash[:error] = "User not found"
        return redirect_to twitch_users_path
      end

      authorize @user
    end

    def set_vod_privacy_property
      @vod_privacy_property = UserVodPrivacyProperty.find(@user.id)
    end

    def set_vod_property
      @vod_property = UserVodProperty.find(@user.id)
    end

    def audit_user_vod_property_update(current_props:, new_props:)
      update_attributes = [
        :hide_archives,
        :hide_archives_enabled,
        :save_vods_forever,
        :skip_upload_moderation,
        :can_upload_unlimited,
        :skip_muting
      ]

      changes = []
      update_attributes.each do |key|

        symbol = key.to_s
        old_value = current_props[symbol]
        new_value = new_props[symbol]

        if old_value == new_value
          next
        end

        change = History::ChangeSet.new(
          attribute: symbol,
          old_value: old_value,
          new_value: new_value
        )
        changes.push(change)

      end

      return if changes.length.zero?

      audit = History::Audit.new(
        action: History::AUDIT_ACTION_UPDATE,
        user_type: History::AUDIT_LDAP_USER,
        user_id: current_user.ldap_login,
        resource_type: History::AUDIT_TWITCH_USER,
        resource_id: @user.id,
        changes: changes
      )

      History::AddAudit.add(audit)
    end
  end
end
