require 'date'

module OauthClients
  class Item < OauthClients::Base
    self.primary_attribute = :client_id

    attributes :id, :client_id, :owner_id, :client_secret, :redirect_uri,
               :category, :oauth_category_id, :name, :password_grant_allowed,
               :should_expire_tokens, :allow_embed, :is_fuel_client, :hidden,
               :is_first_party, :created_at, :updated_at

    validates :id, :client_id, presence: true

    AUDIT_LDAP_USER = "ldap_user"

    def initialize(opts = {})
      opts.each do |k, v|
        if self.respond_to? k
          self.send("#{k}=", v)
        end
      end
    end

    def self.all(options = {})
      @items = []
      opts_hash = options.to_hash
      opts_hash[:show_hidden] = true
      response = get "/clients", params: opts_hash

      return @items, '' unless response.success?
      response.body['clients'].each do |client|
        @items << from_json(client)
      end
      cursor = response.body['next_cursor']
      return @items, cursor
    end

    def self.find(id, options = {})
      response = get "/clients/#{id}"
      return from_errors(response.body) unless response.success?
      from_json(response.body)
    end

    def self.create(admin, options = {})
      body = JSON.dump({
       name: options[:name],
       owner_id: options[:owner_id].to_i,
       redirect_uri: options[:redirect_uri],
       category: options[:category],
       oauth_category_id: 1
     })
      response = post "/clients", body: body, params: {user_id: admin, user_type: AUDIT_LDAP_USER}
      return from_errors(response.body) unless response.success?

      from_json(response.body)
    end

    def update(admin)
      return false unless valid?
      response = put "/clients/#{primary_attribute_value}", body: update_attributes.to_json, params: {user_id: admin, user_type: AUDIT_LDAP_USER}

      self.errors = response.body unless response.success?

      response.success?
    end

    def destroy(admin)
      response = delete "/clients/#{primary_attribute_value}", params: {user_id: admin, user_type: AUDIT_LDAP_USER}

      self.errors = response.body unless response.success?

      response.success?
    end

    def update_attributes
      attributes_to_slice = [
        :name, :owner_id, :redirect_uri, :category, :hidden,
        :is_first_party, :is_fuel_client, :password_grant_allowed, :should_expire_tokens,
        :allow_embed
      ]
      self.attributes.slice(*attributes_to_slice)
    end

    def self.from_json(json)
      permissions = json.delete('permissions')
      settings = json.delete('settings')
      json = json.merge(permissions).merge(settings)
      Item.new json
    end

    def created_at
      time = @created_at.present? ? DateTime.rfc3339(@created_at) : DateTime.new(0)
      time.to_formatted_s(:long_ordinal)
    end

    def updated_at
      time = @updated_at.present? ? DateTime.rfc3339(@updated_at) : DateTime.new(0)
      time.to_formatted_s(:long_ordinal)
    end
  end
end
