module PayoutEntities
  class PayoutEntity < PayoutEntities::Base
    attributes :payout_entity_id,
      :owner_channel_id,
      :tax_withholding_rate,
      :is_legacy,
      :payout_status,
      :channel

    self.primary_attribute = :payout_entity_id

    def self.find(id, options: {}, find_by_channel_id: false)
      if find_by_channel_id

        begin
          response = get "/payout/#{id}/payout_entities", headers: ldap_headers(options)
        rescue Faraday::ClientError => e
          from_errors(['Payout Entity search failed', e])
        else
          if response.success?
            payout_entities = response.body["payout_entities"]

            return from_attributes({}) if payout_entities.nil?

            from_attributes(payout_entities.first)
          elsif response.body.present? && response.body["message"].present?
            from_errors(response.body["message"])
          else
            from_errors("Payout Entity search timed out")
          end
        end
      else
        response = get "/payout/entity/#{id}"

        if response.success?
          from_attributes(response.body)
        elsif response.body.present?
          from_errors(response.body["message"])
        end
      end
    end

    def self.find_all_owned_by_channel(channel_id, options)
      response = get "/payout/#{channel_id}/payout_entities", headers: ldap_headers(options)

      return [] unless response.success?
      return response.body["payout_entities"]
    end

    def self.payout_type(channel_id)
      Ripley::BatchGetPayoutType.call([channel_id.to_s])[channel_id.to_s]
    end

    def self.workflow_events_for_channel(channel_id)
      response = get "/payout/#{channel_id}/workflow_events"

      return {} unless response.success?
      result = response.body["workflow_events"].sort_by { |event| event["timestamp"] }.reverse!
      result = result.each do |event|
        event["timestamp"] = format_timestamp(event["timestamp"])
      end
    end

    def self.workflow_events_for_payout_entity(payout_entity_id, options)
      response = get "/payout/entity/#{payout_entity_id}/workflows", headers: ldap_headers(options)

      return [] unless response.success?
      result = response.body["workflow_events"].sort_by { |event| event["timestamp"] }.reverse!
      result = result.each do |event|
        event["timestamp"] = format_timestamp(event["timestamp"])
      end
    end

    def self.payout_attributions(payout_entity_id, options)
      response = get "/payout/entity/#{payout_entity_id}/attributions", headers: ldap_headers(options)

      return [] unless response.success?
      return response.body["payout_attributions"]
    end

    def self.live_payout_entity_id(channel_id)
      response = get "/payout/#{channel_id}/live_payout_entity"

      return {} unless response.success?
      return response.body["payout_entity_id"]
    end

    private

    def self.format_timestamp(string)
      Time.parse(string).utc.strftime("%Y-%m-%d %H:%M:%S UTC") rescue ""
    end
  end
end
