module Subscriptions
  class TicketProduct < Subscriptions::Base
    self.service_endpoint = Settings.valhalla.endpoint
    self.primary_attribute = :id

    attr_accessor :subber, :prefix, :prefix_state

    attributes :id,
      :emoticons,
      :emoticon_set,
      :availability_state,
      :bitrate_access_string,
      :channel,
      :default_price,
      :desc_checkout,
      :desc_invoice,
      :ext_product_id,
      :ext_product_name,
      :ext_product_description,
      :features,
      :hide_ads,
      :interval_number,
      :interval_unit,
      :name,
      :paywall_css,
      :paywall_html,
      :recurring,
      :rev_share,
      :short_name,
      :ticket_access_start,
      :ticket_access_end,
      :ticket_type,
      :ticket_product_owner_type,
      :created_on,
      :updated_on,
      :price_in_cents,
      :should_show_mass_dnr,
      :should_show_create_tiers,
      :_links

    def self.search(filter)
      channel_login = filter[:login]
      channel_id = filter[:channel_id]
      short_name = filter[:short_name]

      return [] if channel_login.blank? && channel_id.blank? && short_name.blank?

      query = {}
      query[:channel_login] = channel_login if channel_login
      query[:channel_id] = channel_id if channel_id
      query[:short_name] = short_name if short_name
      query[:exact_match] = !!filter[:exact_match]

      response = get "/admin_ticket_products", params: query
      return [] unless response.success? && response.body.is_a?(Hash)

      response.body["results"].to_a.map do |ticket_product|
        from_attributes(ticket_product)
      end
    end

    def self.find(id)
      response = get "/admin_ticket_products/#{id}"

      if response.success? && response.body.is_a?(Hash)
        from_attributes(response.body)
      else
        from_errors(response.body)
      end
    end

    def self.find_with_pending_emoticon_prefixes
      response = get "/admin_ticket_products/get/pending_emoticon_prefixes"
      return [] unless response.success?

      response.body.to_a.map do |ticket_product|
        from_attributes(ticket_product)
      end
    end

    def update(params)
      options = {
        name: params[:name],
        rev_share: params[:rev_share],
        interval_unit: params[:interval_unit],
        ticket_access_start: params[:ticket_access_start],
        ticket_access_end: params[:ticket_access_end],
        desc_invoice: params[:desc_invoice],
        desc_checkout: params[:desc_checkout],
        paywall_html: params[:paywall_html],
        paywall_css: params[:paywall_css],
        bitrate_access_string: params[:bitrate_access_string],
        recurring: params[:recurring]
      }

      response = patch "/admin_ticket_products/#{params[:id]}", {
        body: options.to_json,
        headers: {"content-type" => "application/json"}
      }

      unless response.success? && response.body.is_a?(Hash)
        errors.add(:base, response.body["error"])
      end

      response
    end

    def deactivate(id)
      response = patch "/admin_ticket_products/#{id}/deactivate"

      unless response.success?
        errors.add(:base, response.body["error"])
      end

      response
    end

    def reactivate(id)
      response = patch "/admin_ticket_products/#{id}/reactivate"

      unless response.success?
        errors.add(:base, response.body["error"])
      end

      response
    end

    def grant(id, user_name)
      options = {
        user_login: user_name,
        id: id
      }

      response = post "admin_ticket_products/#{id}/grant", {
        body: options.to_json,
        headers: {"content-type" => "application/json"}
      }

      unless response.success?
        errors.add(:base, response.body["error"])
      end

      response
    end

    def mass_dnr(email, id)
      options = {
        admin_email: email,
        product_id: id
      }

      response = post "admin_ticket_products/#{id}/mass_dnr", {
        body: options.to_json,
        headers: {"content-type" => "application/json"}
      }

      unless response.success?
        errors.add(:base, response.body["error"])
      end

      response
    end

    def self.create_additional_tiers(id)
      response = post "admin_ticket_products/#{id}/create_additional_tiers"

      response
    end

    def login
      if @channel
        @channel["login"]
      else
        ""
      end
    end

    def channel_id
      @channel ? @channel["id"] : ""
    end

    def emoticon_set_id
      @emoticon_set ? @emoticon_set["id"] : ""
    end

    def ticket_access_start_string
      @ticket_access_start
    end

    def ticket_access_end_string
      @ticket_access_end
    end

    def ticket_access_start
      Date.parse(@ticket_access_start) rescue nil
    end

    def ticket_access_end
      Date.parse(@ticket_access_end) rescue nil
    end

    def created_on
      Date.parse(@created_on) rescue nil
    end

    def updated_on
      Date.parse(@updated_on) rescue nil
    end

    def bitrate_access_string
      if @bitrate_access_string
        @bitrate_access_string.join(",")
      else
        []
      end
    end

    def revenue_configs
      Substwirp::RevenueConfig.get_configs_for_product_id(@id)
    end
  end
end
