module Vods
  class Vod < Vods::Base
    include ActiveModel::Serialization

    self.primary_attribute = :id

    attributes :id, :owner_id, :broadcast_id, :broadcast_type, :title, :game,
      :started_on, :uri, :offset, :duration, :status, :show_formats, :created_at,
      :updated_at, :views, :tag_list, :thumbnail_templates, :description,
      :broadcaster_software, :deleted, :created_by, :language, :delete_at,
      :manifest, :viewable_at, :origin, :resolutions

    alias formats show_formats
    alias tags tag_list
    alias deleted_at delete_at

    def self.find(id)
      response = twirp_client.internal_get_vod_by_i_d_include_banned_users(vod_id: id)

      if response.error
        return nil if response.error.code == :not_found
        raise response.error.msg
      end

      self.from_twirp_object(response.data.vod)
    end

    def mute_info
      return if id.blank?

      @mute_info ||= begin
        response = self.class.twirp_client.internal_get_vod_mute_info(vod_id: id.to_s)
        raise response.error.msg if response.error
        response.data.mute_info
      end
    end

    def vod_appeal
      return if mute_info.vod_appeal.blank?
      @vod_appeal ||= Vods::VodAppeal.from_twirp_object(mute_info.vod_appeal)
    end

    def track_appeals
      @track_appeals ||= begin
        mute_info.track_appeals.map do |track_appeal|
          Vods::TrackAppeal.from_twirp_object(track_appeal)
        end
      end
    end

    def audible_magic_responses
      @audible_magic_responses ||= begin
        mute_info.audible_magic_responses.map do |audible_magic_response|
          Vods::AudibleMagicResponse.from_twirp_object(audible_magic_response).tap do |amr|
            amr.track_appeal = track_appeals.find { |track_appeal| track_appeal.audible_magic_response_id == amr.id }
          end
        end
      end
    end

    def thumbnails
      thumbnail_templates.map(&:to_h)
    end

    def format_names
      grouped_formats_for(:display_name)
    end

    def fpses
      grouped_formats_for(:fps)
    end

    def bitrates
      grouped_formats_for(:bitrate)
    end

    def codecs
      grouped_formats_for(:codecs)
    end

    def grouped_formats_for(key)
      formats.to_h.each_with_object({}) do |(k, v), h|
        h[k] = v[key]
        h
      end
    end

    def owner
      return if owner_id.blank?
      @owner ||= Twitch::User.find(owner_id)
    end

    def created_by_user
      return if created_by.blank?
      @created_by_user ||= Twitch::User.find(created_by)
    end

    def duration
      @duration.to_i
    end

    def destroy
      raise "Call soft_delete or hard_delete"
    end

    def soft_delete
      response = self.class.twirp_client.soft_delete_vods(vod_ids: [primary_attribute_value])
      raise response.error.msg if response.error
    end

    def hard_delete
      response = self.class.twirp_client.hard_delete_vods(vod_ids: [primary_attribute_value])
      raise response.error.msg if response.error
    end

    def undelete
      response = self.class.twirp_client.undelete_vods(vod_ids: [primary_attribute_value])
      raise response.error.msg if response.error
    end
  end
end
