Rails.application.routes.draw do

  redirect_with_error = lambda do |message, destination|
    redirect do |_, req|
      req.flash[:error] = message
      destination # redirect to the main
    end
  end

  namespace :content_tag do
    resources :channel_tags do
      get :delete_channel_tag, on: :member
      patch :add_channel_tag, on: :member
    end
  end
  namespace :payments do
    resources :purchase_refunds
  end
  namespace :zuma do
    resources :user_blocks, only: [:index]
    resources :is_blocking, only: [:index]
  end
  namespace :user_rewards do
    resources :bits_badge_tiers, only: [:index]
    post "grant_unlocked_bits_badge_tier_rewards", to: "bits_badge_tiers#grant_unlocked_rewards"

    resources :hype_trains, only: [:index]
    post "grant_hype_train_rewards", to: "hype_trains#grant_rewards", as: :grant_hype_train_rewards
  end
  namespace :games do
    resources :add_category_tags
  end
  namespace :graffiti do
    resources :edit_tags
  end
  namespace :clue do
    resources :chat_properties, only: [:index, :edit, :update]
  end
  namespace :dmca do
    get "takedowns_received", controller: "takedowns"
    get "takedowns_issued", controller: "takedowns"
  end
  namespace :dmca do
    resources :users
  end
  namespace :rewards do
    resources :campaigns do
      collection do
        post :invalidate_cache
      end
    end
    resources :objectives
    resources :milestones
    resources :rewards
  end
  namespace :blocked_ips do
    resources :items
  end
  namespace :staff do
    resources :members do
      resources :activity, only: [:index]
    end
  end
  namespace :games do
    resources :add_localizations, only: [:index, :create]
    resources :add_game_banners, only: [:index, :show, :update], constraints: { id: /.+/ }
    resources :add_aliases, only: [:index, :create]
    resources :add_non_giantbomb_games, only: [:index, :create]
    resources :add_categories, only: [:index, :create]
    resources :add_boxarts, only: [:index, :show, :update], constraints: { id: /.+/ }
  end
  namespace :drops do
    resources :developers, only: [:index, :show]
    resources :users, only: [:index]
  end
  namespace :bits do
    resources :users do
      post :add_bits
      post :add_bits_for_campaign
      post :remove_bits
      post :set_ban_status
      post :set_opt_out_status
      post :moderate_cheermote
      get :download_cheermotes
      resources :transactions, only: [:index]
    end
    resources :jobs, only: [:index] do
      get :get_input
      get :get_options
      get :get_output
      post :create_bulk_entitle, on: :collection
      post :create_bulk_entitle_by_campaign, on: :collection
    end
    resources :types, only: [:index, :new, :create, :edit, :update], constraints: { id: /.+/ }
    resources :products, only: [:index, :new, :create, :update], constraints: { id: /.+/ }
    get "products/:platform/:id/edit", to: "products#edit", as: :edit_product, constraints: { id: /.+/ }
    put "products/:platform/:id/update", to: "products#update", as: :update_product, constraints: { id: /.+/ }

    get "badges", to: "badges#index"
    get ":user_id/badges", to: "badges#all", as: :user_badges
    post ":user_id/update_badge_tiers", to: "badges#update_badge_tiers", as: :update_badge_tiers
    post ":user_id/moderate_badge_tiers", to: "badges#moderate_badge_tiers", as: :moderate_badge_tiers

    resources :gem_notifications, only: [:index, :create] do
      post :activate
      post :deactivate
    end
    resources :actuals, only: [:index] do
      post :configure_actuals, on: :collection
    end
  end
  namespace :keys do
    resources :pools, only: [:index, :create] do
      post :activate
      post :deactivate
    end
    resources :batches, only: [:index, :create] do
      post :activate
      post :deactivate
    end
    resources :product_types, only: [:index, :create]
    resources :products, only: [:index, :create]
    resources :reports, only: [:index, :create] do
      post :download
    end
    resources :configurations, only: [:index, :create] do
      post :delete
    end
    resources :handler_groups, only: [:index, :create]
  end
  namespace :transactions do
    resources :entries, only: [:index] do
      post :download
    end
  end
  namespace :revenue_reporting do
    resources :reports, only: [:index, :create] do
      collection do
        post :tax_withholding_rates
      end
    end
    resources :payout_details, only: [:index]
    resources :fraud_reports, only: [:index]
    resources :block_payouts, only: [:index]
  end

  namespace :gift_cards do
    resources :redemption, only: [:index]

    resources :balance, only: [:index] do
      post :modify
      post :wallet_status
    end
  end

  namespace :payments do
    resources :purchase_profiles, only: [:index, :show] do
      post :cancel
      post :do_not_renew
      post :sync_ticket_access
    end
    resources :purchase_payments, only: [:show]
    resources :xsolla_plans, only: [:index, :new, :create]
    resources :purchase_orders, only: [:index, :show] do
      post :cancel
      post :do_not_renew
      post :label_fraud
      post :refund_fraud
      post :mark_refunded_by_finance
      post :refund
    end
  end
  # We are not using resources because we require dynamic routes with defaults for some arities.
  namespace :promotions do
    resources :backfill_channels
    resources :scheduled_vods
  end

  namespace :oauth_clients do
    resources :items
  end

  namespace :boxart do
    resources :games, only: [:index, :show, :update]
  end

  namespace :twitch do
    # friendly errors for Oracle removal
    [:events, :featured_event_channels].each do |ns|
      namespace ns do
        get('(*any)', to:
          redirect_with_error.call(
            "Events support is deprecated. Please contact #dashboard-analytics on Slack if this impacts you.",
            'twitch/users'
          ))
      end
    end
    resources :reservations, only: [:index, :show, :destroy, :new, :create, :update]
    resources :admins, only: [:index]
    resources :team_memberships, param: :slug, only: :destroy

    resources :teams, param: :name do
      get 'search', on: :collection
      get 'delete', on: :member

      resources :members, controller: 'team_members', only: :index
      resources :memberships, controller: 'team_memberships', only: :create
    end

    resources :users, only: [:index, :show, :edit, :update, :destroy] do
      member do
        delete :disable, controller: "two_factor"
        post :undelete
        delete :soft_delete
        get :hard_delete
        post :compare_geo_ip
        post :disable_twitchguard
      end

      resource :platform_connections, only: [:destroy]
      resource :impersonate, only: [:create]
      resource :report_clips, only: [:destroy]
      resource :roles, only: [:edit, :update]
      resource :suspensions, only: [:destroy]
      resource :validated_non_partner, only: [:edit, :update], controller: "validated_non_partner"

      resource :rename, only: [:new, :create], controller: "user_renames" do
        member do
          get :new_override
          post :create_override
        end
      end

      resources :channels, only: [:edit, :update]

      resource :channel_property, controller: "user_channel_properties", only: [:edit, :update] do
        member do
          put :regenerate_stream_key
          post :add_authorized_stream_key
          post :remove_authorized_stream_key
          post :disconnect
        end
      end

      resource :passwords, only: [], controller: "passwords" do
        member do
          post :force_reset
          post :change_password
        end
      end

      resource :chat, only: [], controller: "chat" do
        member do
          post :disconnect_chat_sessions
        end
      end

      resource :sessions, only: [], controller: "sessions" do
        member do
          post :delete_all_cookies
        end
      end

      resource :oauth, only: [], controller: "oauth" do
        member do
          post :delete_all_authorizations
        end
      end
    end

    scope :users do
      namespace :platform_connections do
        post :query
      end

      namespace :accounts do
        post :reset
      end
    end

  end

  namespace :ads do
    resources :auto_ads, only: [:edit, :update]
    resources :settings, only: [:edit, :update]
    resources :revenue_configs, only: [:create, :destroy] do
      collection do
        get 'new_bulk'
        post 'create_bulk'
      end
    end
    resources :users, only: [] do
      resources :revenue_configs, only: [:new]
    end
  end

  namespace :beef_cake do
    resources :users, id: /[a-z0-9\-\._]+/i do
      put :reactivate, on: :member
    end

    resources :roles do
      resources :attached_permissions
      resources :bulk_edit
      post :remove_user
    end
    resources :permissions
    resources :memberships
  end

  namespace :vods do
    resources :vods, only: [:index, :show, :destroy] do
      member do
        post :undestroy
        post :download
      end
    end
    resources :channels, only: [:index]
    resources :track_appeals, only: [] do
      member do
        post :resolve
      end
    end
    resources :user_property, only: [:edit, :update]
    resources :privacy_property, only: [:edit, :update]
  end
  namespace :vod_appeals do
    resources :appeals, only: [:index] do
      member do
        post :resolve
      end
    end
  end

  namespace :attachments do
    resources :hosted_images, only: [:index, :create]
  end

  namespace :notifications do
    resources :items do
      collection do
        get :markdown
      end
    end
  end

  namespace :subscriptions do
    resources :tokens
    post '/tokens/grant_tokens', to: 'tokens#grant_tokens'
    post '/tokens/get_token_history', to: 'tokens#get_token_history'

    resources :channels, only: [] do
      resource :badge_timeout_status, only: [:update]
      resources :badges, only: [:index, :create, :destroy]
    end

    resources :fix_gap, only: [:update]
    resources :fix_latest, only: [:update]

    resources :tenures do
      collection do
        post :update
        post :override
      end
    end
    post 'tenures/delete_grant', to: 'tenures#delete_grant'
    post 'tenures/upsert_grant', to: 'tenures#upsert_grant'
    post 'tenures/preview_tenure', to: 'tenures#preview_tenure'

    resources :resub_tokens, only: [:index] do
    end
    post 'resub_tokens/allow_replay', to: 'resub_tokens#allow_replay'

    resources :ticket_products, only: [:index, :show, :edit, :update, :new, :create] do
      put :sync_external_ticket_product
      patch :deactivate, to: 'ticket_products#deactivate'
      patch :reactivate, to: 'ticket_products#reactivate'
      post :grant_admin_sub, to: 'ticket_products#grant_admin_sub'
      post :mass_dnr, to: 'ticket_products#mass_dnr'
      post :create_additional_tiers, to: 'ticket_products#create_additional_tiers'
    end

    resources :pending_emoticon_prefixes, only: [:index, :update] do
    end

    resources :emotes, only: [:index, :edit, :update] do
      patch :deactivate, to: 'emotes#deactivate'
      post :upload, to: 'emotes#upload'
    end

    resources :ticket_product_revenue_configs, only: [:index, :edit, :update, :create, :new] do
      get '/new/:ticket_product_id', to: 'ticket_product_revenue_configs#new'
      post '/delete', to: 'ticket_product_revenue_configs#delete'
    end
  end

  namespace :payout_entities do
    resources :workflows, only: [:show, :edit, :update, :destroy]
    resources :attributions, only: [:create, :destroy]
    resources :items, only: [:index, :show]
    resources :invitations, only: [:index, :create, :destroy]
    resources :bulk_invitations, only: [:index, :create]
    resources :activations, only: [:index, :create]
    resources :users, only: [] do
      post :migrate_legacy_partner_workflow, to: 'workflows#migrate_legacy_partner_workflow'
      resources :activations, only: [:new]
      resources :invitations, only: [:new] do
        collection do
          post :remove
        end
      end
    end
  end

  namespace :resque_jobs do
    resources :jobs do
      member do
        post :kill
      end
      collection do
        delete :delete_all
        delete :delete_completed
        delete :delete_failed
      end
    end
    resources :workers, only: [:index]
  end

  namespace :support_tools do
    resources :users, only: [:index, :create]
    resources :account_recovery, only: [:index, :create]
  end

  namespace :history do
    resources :query_audits, only: [:index]
    resources :reports, only: [:index, :create]
    resources :deletions, only: [:index, :create]
  end

  root "pages#index"

  get "/login", to: redirect("/auth/ping_federate_oauth2"), as: :login
  get "/login_status", to: "sessions#login_status", as: :login_status
  get "/twofa", to: "sessions#twofa", as: :twofa
  post "/twofa_callback", to: "sessions#twofa_callback", as: :twofa_callback
  delete "/logout", to: "sessions#destroy", as: :logout

  namespace :banned_words do
    resources :items
  end

  namespace :tools do
    namespace :ip_resolver do
      post 'resolve', to: 'ip_resolver'
      get 'resolve', to: 'ip_resolver'
      get 'show', to: 'ip_resolver'
    end
    namespace :translator do
      root action: :show, as: :root
      get 'show', to: 'translator'
      get 'translate', to: redirect(path: 'tools/translator')
    end
  end

  get '/auth/:provider/callback', to: 'sessions#oauth2_callback', as: :oauth2_callback
  get '/auth/failure', to: "sessions#auth_failure", as: :auth_failure

  post "/tracking/heartbeat", to: "tracking#heartbeat", as: :tracking_heartbeat

  get "/elb-health", to: "health#status"

  namespace :graffiti do
    resources :create_tags, only: [:index, :create]
  end
end
