require 'rails_helper'

RSpec.describe Ads::RevenueConfigsController do
  include_context 'mocked permissions'

  let(:permission) { ['ad_revenue_configs'] }
  let(:user_id) { "27231752" }
  let(:config_id) { "1" }
  let(:dummy_config) { Ads::RevenueConfig.new revenue_config }
  let(:revenue_config) do
    {
      "created_at" => "2012-04-02 23:25:32",
      "flat_rate_cpm_in_cents" => "350",
      "id" => "5113",
      "rev_split_percent" => "",
      "start_date" => "2012-03-01 08:00:00",
      "t1_flat_rate_cpm_in_cents" => "",
      "t1_rev_split_percent" => "",
      "t2_flat_rate_cpm_in_cents" => "",
      "t2_rev_split_percent" => "",
      "updated_at" => "2012-04-02 23:25:32",
      "user_id" => user_id
    }
  end
  describe 'create_bulk' do
    describe 'permissions' do
      let(:action) { :create_bulk }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Ads::RevenueConfig).to \
            receive(:new) \
            .with(ActionController::Parameters.new(revenue_config)) \
            .and_return(dummy_config)
          expect(dummy_config).to receive(:save).and_return(true)
          post action, params: { ads_revenue_config: revenue_config }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          post action, params: { ads_revenue_config: revenue_config }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'create' do
    describe 'permissions' do
      let(:action) { :create }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Ads::RevenueConfig).to \
            receive(:new) \
            .with(ActionController::Parameters.new(revenue_config)) \
            .and_return(dummy_config)
          expect(dummy_config).to receive(:save).and_return(true)
          post action, params: { ads_revenue_config: revenue_config }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          post action, params: { ads_revenue_config: revenue_config }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'destroy' do
    let(:action) { :destroy }

    context 'service returns success' do
      include_context 'an authorized user'

      before do
        expect(Ads::RevenueConfig).to receive(:new).with({ id: config_id })\
          .and_return(dummy_config)
        expect(dummy_config).to receive(:destroy).and_return(true)
        delete action, params: { id: config_id }
      end

      it_behaves_like 'it redirects the user and does not display an error'
    end

    context 'service returns failure' do
      include_context 'an authorized user'

      before do
        expect(Ads::RevenueConfig).to receive(:new).with({ id: config_id })\
          .and_return(dummy_config)
        expect(dummy_config).to receive(:destroy).and_return(false)
        delete action, params: { id: config_id }
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    describe 'permissons' do

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Ads::RevenueConfig).to receive(:new).with( { id: config_id } )\
            .and_return(dummy_config)
          expect(dummy_config).to receive(:destroy).and_return true
          delete action, params: { id: config_id }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          delete action, params: { id: config_id }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end
end
