require 'rails_helper'

RSpec.describe BeefCake::AttachedPermissionsController do
  include_context 'mocked permissions'

  let(:permission) { 'admin_permissions' }
  let(:id) { 'my-id' }
  let(:role_id) { 'my-role-id' }
  let(:permission_name) { 'my-permission-name' }
  let(:role_name) { 'my-role-name' }
  let(:attached_permission) do
    BeefCake::AttachedPermission.new(
      role: BeefCake::Role.new(name: role_name),
      permission: BeefCake::Permission.new(name: permission_name)
    )
  end

  describe 'edit' do
    let(:action) { :edit }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user'

      before(:each) do
        expect(BeefCake::AttachedPermission).to receive(:find)
          .with(role_id: role_id, attached_permission_id: id)
          .and_return(attached_permission)
        get action, :params => {:id => id, role_id: role_id}
      end

      it_behaves_like 'it returns a success response'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'

      before do
        expect(BeefCake::AttachedPermission).to receive(:find)
          .with(role_id: role_id, attached_permission_id: id)
          .and_return(attached_permission)
        get action, :params => {:id => id, role_id: role_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'update' do
    let(:action) { :update }
    let(:scope_s) { ['AFFILIATE'] }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user'

      before(:each) do
        expect(BeefCake::AttachedPermission).to receive(:find)
          .with(role_id: role_id, attached_permission_id: id)
          .and_return(attached_permission)
        expect(attached_permission).to receive(:update_scope)
          .with(scope_s)
        get action, params: {
          id: id,
          role_id: role_id,
          beef_cake_attached_permission: {
            payout_entity_scope: scope_s
          }
        }
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'

      before do
        expect(BeefCake::AttachedPermission).to receive(:find)
          .with(role_id: role_id, attached_permission_id: id)
          .and_return(attached_permission)
        get action, :params => {:id => id, role_id: role_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'destroy' do
    let(:action) { :destroy }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user'

      before(:each) do
        expect(BeefCake::AttachedPermission).to receive(:find)
          .with(role_id: role_id, attached_permission_id: id)
          .and_return(attached_permission)
        expect(attached_permission).to receive(:delete)
        get action, :params => {:id => id, role_id: role_id}
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'

      before do
        expect(BeefCake::AttachedPermission).to receive(:find)
          .with(role_id: role_id, attached_permission_id: id)
          .and_return(attached_permission)
        get action, :params => {:id => id, role_id: role_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end
end
