require 'rails_helper'

RSpec.describe BeefCake::BulkEditController do
  include_context 'mocked permissions'
  let(:permission) { 'admin_permissions' }
  let(:role_id) { 'ROLEID' }
  let(:role) { BeefCake::Role.from_attributes(id: role_id) }

  before(:each) do
    expect(BeefCake::Role).to receive(:find).with(role_id).and_return(role)
  end

  describe 'index' do
    let(:action) { :index }
    let(:params) { {role_id: role_id} }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user'

      before do
        get action, params
      end

      after do
        expect(@controller.view_assigns['realms']).to eq(BeefCake::Realm.all)
        expect(@controller.view_assigns['role']).to eq(role)
      end

      it_behaves_like 'it returns a success response'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'

      before do
        get action, params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'create' do
    let(:batch_action) { 'BATCH_ACTION' }
    let(:realm) { BeefCake::Realm::TWITCH_LDAP }
    let(:user_id) { 'my_user_id' }
    let(:action) { :create }
    let(:params) do
      {
        batch_action: batch_action,
        realm_id: realm.id,
        role_id: role_id,
        user_ids: user_id
      }
    end

    shared_context 'after sending the request' do
      before(:each) { post action, params }
    end

    context 'a user with appropriate permissions' do
      include_context 'an authorized user'

      let(:bulk_action_result) { true }

      before(:each) do
        expect(role).to receive(role_bulk_action)
          .with(realm, [user_id])
          .and_return(bulk_action_result)
          .once
      end

      shared_context 'with validation failures' do
        let(:bulk_action_result) { false }
        before(:each) { role.errors.add(:user_ids, 'my error') }
      end

      shared_examples 'it redirects back with errors' do
        it 'has the same location' do
          expect(response.location).to eq(response.request.url)
        end
        it 'has shows an error' do
          expect(flash[:error]).to be_present
        end
      end

      context 'with Add as the batch action' do
        let(:batch_action) { 'Add' }
        let(:role_bulk_action) { :bulk_add_realm_users }

        context 'on success' do
          include_context 'after sending the request'
          it_behaves_like 'it redirects the user and does not display an error'
        end

        context 'on failure' do
          include_context 'with validation failures'
          include_context 'after sending the request'
          it_behaves_like 'it redirects back with errors'
        end
      end

      context 'with Remove as the batch action' do
        let(:batch_action) { 'Remove' }
        let(:role_bulk_action) { :bulk_remove_realm_users }

        context 'on success' do
          include_context 'after sending the request'
          it_behaves_like 'it redirects the user and does not display an error'
        end

        context 'on failure' do
          include_context 'with validation failures'
          include_context 'after sending the request'
          it_behaves_like 'it redirects back with errors'
        end
      end
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'
      include_context 'after sending the request'
      it_behaves_like 'it redirects the user and displays an error'
    end
  end
end
