require 'rails_helper'

RSpec.describe BeefCake::PermissionsController do

  include_context 'mocked permissions'

  let(:permission) { 'admin_permissions' }

  describe 'index' do
    describe 'permissions' do
      let(:action) { :index }

      context 'a user with appropriate permissions' do
        include_context 'user with atleast one permission'

        before do
          expect(BeefCake::Permission).to receive(:all).and_return([])
          get action
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'user with no permission'

        before do
          get action
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'update' do
    describe 'permissions' do
      let(:action) { :update }
      let(:dummy_item) { BeefCake::Permission.new }
      let(:params) do
        {
          id: 'foo',
          beef_cake_permission: {
            description: 'very good permission'
          }
        }
      end

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'
        let(:save_result) { true }

        before do
          expect(BeefCake::Permission).to receive(:find).with('foo').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          expect(dummy_item).to receive(:save).and_return(save_result)
          patch action, params: params
        end

        it_behaves_like 'it redirects the user and displays a success message'

        context 'update error' do
          let(:save_result) { false }
          let(:dummy_item) do
            perm = BeefCake::Permission.new
            perm.errors.add(:id, message: 'a save error')
            perm
          end

          it_behaves_like 'it returns a success response and displays a danger'
        end
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('foo').and_return(dummy_item)
          patch action, params: params
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

    end

  end

  describe 'show' do
    describe 'permissions' do
      let(:action) { :show }
      let(:dummy_item) { BeefCake::Permission.new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('foo').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          get action, :params => {:id => 'foo'}
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('foo').and_return(dummy_item)
          get action, :params => {:id => 'foo'}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'new' do
    describe 'permissions' do
      let(:action) { :new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          get action, :params => {:beef_cake_permission => {:name => 'very good permission'}}
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          get action, :params => {:beef_cake_permission => {:name => 'very good permission'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'create' do
    describe 'permissions' do
      let(:action) { :create }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect_any_instance_of(BeefCake::Permission).to receive(:save).and_return(true)
          post action, :params => {:beef_cake_permission => {:name => 'very good permission'}}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          post action, :params => {:beef_cake_permission => {:name => 'very good permission'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'edit' do
    describe 'permissions' do
      let(:action) { :edit }
      let(:dummy_item) { BeefCake::Permission.new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('lel').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          get action, :params => {:id => 'lel'}
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('lel').and_return(dummy_item)
          get action, :params => {:id => 'lel'}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'update' do
    describe 'permissions' do
      let(:action) { :update }
      let(:dummy_item) { BeefCake::Permission.new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('lel').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          expect(dummy_item).to receive(:save).and_return(true)
          post action, :params => {:id => 'lel', :beef_cake_permission => {:name => 'very good permission'}}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('lel').and_return(dummy_item)
          post action, :params => {:id => 'lel', :beef_cake_permission => {:name => 'very good permission'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'destroy' do
    describe 'permissions' do
      let(:action) { :destroy }
      let(:dummy_item) { BeefCake::Permission.new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('lel').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          expect(dummy_item).to receive(:destroy).and_return(true)
          post action, :params => {:id => 'lel'}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Permission).to receive(:find).with('lel').and_return(dummy_item)
          post action, :params => {:id => 'lel'}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end
end
