require 'rails_helper'

RSpec.describe BeefCake::RolesController do

  include_context 'mocked permissions'

  let(:permission) { 'admin_permissions' }

  describe 'index' do
    describe 'permissions' do
      let(:action) { :index }

      context 'a user with appropriate permissions' do
        include_context 'user with atleast one permission'

        before do
          expect(BeefCake::Role).to receive(:all).and_return([])
          get action
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'user with no permission'

        before do
          get action
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'show' do
    describe 'permissions' do
      let(:action) { :show }
      let(:dummy_item) { BeefCake::Role.new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Role).to receive(:find).with('foo').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          get action, :params => {:id => 'foo'}
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Role).to receive(:find).with('foo').and_return(dummy_item)
          get action, :params => {:id => 'foo'}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

    end

  end

  describe 'update' do
    describe 'permissions' do
      let(:action) { :update }
      let(:dummy_item) { BeefCake::Role.new }
      let(:params) { {id: 'foo', beef_cake_role: {name: 'new name'} } }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Role).to receive(:find).with('foo').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          expect(dummy_item).to receive(:save).and_return(true)
          patch action, params: params
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Role).to receive(:find).with('foo').and_return(dummy_item)
          patch action, params: params
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

    end

  end

  describe 'new' do
    describe 'permissions' do
      let(:action) { :new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          get action, :params => {:beef_cake_role => {:name => 'very good role'}}
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          get action, :params => {:beef_cake_role => {:name => 'very good role'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'create' do
    describe 'permissions' do
      let(:action) { :create }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect_any_instance_of(BeefCake::Role).to receive(:save).and_return(true)
          post action, :params => {:beef_cake_role => {:name => 'very good role'}}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          post action, :params => {:beef_cake_role => {:name => 'very good role'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'destroy' do
    describe 'permissions' do
      let(:action) { :destroy }
      let(:dummy_item) { BeefCake::Role.new }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BeefCake::Role).to receive(:find).with('lel').and_return(dummy_item)
          expect(dummy_item).to receive(:persisted?).and_return(true)
          expect(dummy_item).to receive(:destroy).and_return(true)
          post action, :params => {:id => 'lel'}
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BeefCake::Role).to receive(:find).with('lel').and_return(dummy_item)
          post action, :params => {:id => 'lel'}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'remove_user' do
    let(:action) { :remove_user }
    let(:role_id) { 'ROLE_ID' }
    let(:user_id) { 'USER_ID' }
    let(:dummy_item) { BeefCake::Role.from_attributes(id: role_id) }
    let(:params) { {role_id: role_id, user_id: user_id, id: role_id} }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user'

      before do
        expect(BeefCake::Role).to receive(:find).with(role_id).and_return(dummy_item)
        expect(dummy_item).to receive(:remove_user).with(user_id).and_return(true)
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'

      before do
        expect(BeefCake::Role).to receive(:find).with(role_id).and_return(dummy_item)
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end
end
