require 'rails_helper'

RSpec.describe Bits::BadgesController do
  include_context 'mocked permissions'

  let(:permission) { ['user_bits', 'moderate_bits_badges'] }
  let(:test_user_id) { '123' }
  let(:test_twitch_user) { Twitch::User.new(id: test_user_id) }

  class SuccessResponse
    attr_accessor :body
    def initialize(options = {})
      @body = options[:body]
    end

    def success?
      true
    end
  end

  describe '#index' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        get :index
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      let(:permission) { ['user_bits', 'user_bits'] }

      before do
        get :index
      end

      it_behaves_like 'it returns a success response'
    end
  end

  describe '#all' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        get :all, params: {user_id: test_user_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      let(:permission) { ['user_bits', 'user_bits'] }

      before do
        expect(Twitch::User).to receive(:find).with(test_user_id).and_return(test_twitch_user)
        expect(Bits::Badge).to receive(:all).with(test_user_id).and_return([])
        expect(Badges::Bits).to receive(:global_display).and_return(double(success?: true, status: 200, body: {'versions' => []}))
        get :all, params: {user_id: test_user_id}
      end

      it_behaves_like 'it returns a success response'
    end
  end

  describe '#update_badge_tiers' do
    describe 'permissions' do
      context 'a user without permission' do
        include_context 'an unauthorized user'

        before do
          post :update_badge_tiers, params: { user_id: test_user_id, tiers: Hash[1, 'enabled', 100, 'disabled'] }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

      context 'a user with permission' do
        include_context 'an authorized user'

        let(:permission) { ['user_bits', 'user_bits'] }

        before do
          expect(Twitch::User).to receive(:find).with(test_user_id).and_return(test_twitch_user)
          expect(Bits::Badge).to receive(:all).with(test_user_id).and_return([])
          expect(Bits::Badge).to receive(:update_badge_tiers).with(test_user_id, [{threshold: 1, enabled: true}, {threshold: 100, enabled: false}]).and_return(SuccessResponse.new)
          post :update_badge_tiers, params: { user_id: test_user_id, tiers: Hash[1, 'enabled', 100, 'disabled'] }
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end
    end
  end

  describe '#moderate_badge_tiers' do
    describe 'permissions' do
      images_to_moderate = [
        "1000",
        "5000"
      ]

      titles_to_moderate = [
        "2000"
      ]

      tier_settings = [
        {
          threshold: 2000,
          delete_title: true,
        },
        {
          threshold: 1000,
          delete_image: true,
        },
        {
          threshold: 5000,
          delete_image: true,
        }
      ]

      context 'a user without permission' do
        include_context 'an unauthorized user'

        before do
          post :moderate_badge_tiers, params: { user_id: test_user_id, tiers_with_violating_images: images_to_moderate, tiers_with_violating_title: titles_to_moderate }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

      context 'a user with permission' do
        include_context 'an authorized user'

        let(:permission) { ['user_bits', 'user_bits'] }

        before do
          expect(Twitch::User).to receive(:find).with(test_user_id).and_return(test_twitch_user)
          expect(Bits::Badge).to receive(:all).with(test_user_id).and_return([])
          expect(Bits::Badge).to receive(:update_badge_tiers).with(test_user_id, tier_settings).and_return(SuccessResponse.new)
          post :moderate_badge_tiers, params: { user_id: test_user_id, tiers_with_violating_images: images_to_moderate, tiers_with_violating_title: titles_to_moderate }
        end

        it_behaves_like 'it redirects the user and displays a success message'
      end

    end
  end
end
