require 'rails_helper'

RSpec.describe Bits::JobsController do
  include_context 'mocked permissions'

  let(:permission) { 'bits_jobs' }
  let(:test_job) { Bits::Job.new }

  describe 'index' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        get :index
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      let(:permission) { 'bits_jobs' }

      before do
        expect(Bits::Job).to receive(:all).and_return([])
        expect(Bits::Campaign).to receive(:all).and_return(double(success?: true, status: 200, body: {'bulk_grant_campaigns' => []}))
        get :index
      end

      it_behaves_like 'it returns a success response'
    end
  end

  describe 'create_bulk_entitle' do
    describe 'permissions' do
      context 'a user without permission' do
        include_context 'an unauthorized user'

        before do
          post :create_bulk_entitle, params: {bits_job: {:annotation => 'foo', :bits_amount => '1', :by => 'username_lookup', :input_users => 'foobar'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

      context 'a user with permission' do
        include_context 'an authorized user'

        let(:permission) { 'bits_jobs' }

        context 'sending create_bulk_entitle request' do
          before do
            expect(Bits::Job).to receive(:new).and_return(test_job)
            expect(test_job).to receive(:start_bulk_entitle_job).and_return(double(success?: true))
            post :create_bulk_entitle, params: {bits_job: {:annotation => 'foo', :bits_amount => '1', :by => 'username_lookup', :input_users => 'foobar'}}
          end

          it_behaves_like 'it redirects the user and displays a success message'
        end
      end
    end

    describe 'missing parameters' do
      context 'a request missing bits amount' do
        include_context 'an authorized user'

        let(:permission) { 'bits_jobs' }

        before do
          post :create_bulk_entitle, params: {bits_job: {:annotation => 'foo', :bits_amount => '0', :by => 'username_lookup', :input_users => 'foobar'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

      context 'a request missing annotation' do
        include_context 'an authorized user'

        let(:permission) { 'bits_jobs' }

        before do
          post :create_bulk_entitle, params: {bits_job: {:annotation => '', :bits_amount => '1', :by => 'username_lookup', :input_users => 'foobar'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'create_bulk_entitle_by_campaign' do
    describe 'permissions' do
      context 'a user without permission' do
        include_context 'an unauthorized user'

        before do
          post :create_bulk_entitle_by_campaign, params: {bits_job: {:annotation => 'foo', :bits_amount => '1', :by => 'username_lookup', :input_users => 'foobar'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end

      context 'a user with permission' do
        include_context 'an authorized user'

        let(:permission) { 'bits_jobs' }

        context 'sending create_bulk_entitle_by_campaign request' do
          before do
            expect(Bits::Job).to receive(:new).and_return(test_job)
            expect(test_job).to receive(:start_bulk_entitle_by_campaign_job).and_return(double(success?: true))
            post :create_bulk_entitle_by_campaign, params: {bits_job: {:campaign_id => 'a_campaign', :by => 'username_lookup', :input_users => 'foobar'}}
          end

          it_behaves_like 'it redirects the user and displays a success message'
        end
      end
    end

    describe 'missing parameters' do
      context 'a request missing campaign id' do
        include_context 'an authorized user'

        let(:permission) { 'bits_jobs' }

        before do
          post :create_bulk_entitle_by_campaign, params: {bits_job: {:campaign_id => '', :by => 'username_lookup', :input_users => 'foobar'}}
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end
end
