require 'rails_helper'

RSpec.describe BlockedIps::ItemsController do

  include_context 'mocked permissions'

  let(:permission) { 'blocked_ips' }
  let(:item) { BlockedIps::Item.new item_params }
  let(:item_params) { { ip: "127.0.0.1" } }

  describe 'index' do
    describe 'permissions' do
      let(:action) { :index }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        context 'a index call with no params' do
          before do
            get action
          end

          it_behaves_like 'it returns a success response'
        end

        context `a index call with an specified IP address` do
          before do
            expect(BlockedIps::Item).to \
              receive(:find).with(ActionController::Parameters.new(item_params)) \
              .and_return(item)
            get action, :params => { item: item_params }
          end

          it_behaves_like 'it returns a success response'
        end
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          get action
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end

  describe 'create' do
    describe 'permissions' do
      let(:action) { :create }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect_any_instance_of(BlockedIps::Item).to receive(:save).and_return(true)
          post action, :params => { item: item_params }
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          post action, :params => { item: item_params }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end
  end

  describe 'destroy' do
    describe 'permissions' do
      let(:action) { :destroy }

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(BlockedIps::Item).to \
            receive(:find).with(ActionController::Parameters.new(item_params)) \
            .and_return(item)
          expect_any_instance_of(BlockedIps::Item).to receive(:destroy).and_return(true)
          delete action, :params => { id: item_params[:ip], item: item_params }
        end

        it_behaves_like 'it returns a success response'
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        before do
          expect(BlockedIps::Item).to \
            receive(:find).with(ActionController::Parameters.new(item_params)) \
            .and_return(item)
          delete action, :params => { id: item_params[:ip], item: item_params }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

  end
end
