require "rails_helper"

RSpec.describe Clue::ChatPropertiesController do
  include_context "mocked permissions"

  let(:permission) { "chat_properties" }
  let(:chat_property_id) { 123 }
  let(:initial_chat_property) { Clue::ChatProperty.new(:id => chat_property_id, :chat_require_verified_account => true, :hide_chatlinks => true, :rituals_enabled => true, :r9k_only_chat => true, :subscribers_only_chat => true, :chat_fastsubs => true, :facebook_connect_moderated => true) }

  describe "index" do
    describe "permissions" do
      context "a logged-in user viewing the index" do
        before do
          get :index
        end

        it_behaves_like "it returns a success response"
      end

      context "a logged-in user searching another user's chat properties by ID" do
        before do
          expect(Clue::ChatProperty).to receive(:find).with(123).and_return(initial_chat_property)
          allow(Twitch::User).to receive(:find).with("123").and_return(Twitch::User.new(:id => chat_property_id))
          get :index, params: { user: "123"}
        end

        it_behaves_like "it returns a success response"
      end

      context "a logged-in user searching another user's chat properties by username" do
        before do
          expect(Clue::ChatProperty).to receive(:find).with(123).and_return(initial_chat_property)
          allow(Twitch::User).to receive(:find_by_login).with("user123").and_return(Twitch::User.new(:id => chat_property_id))
          get :index, params: { user: "user123"}
        end

        it_behaves_like "it returns a success response"
      end
    end
  end

  describe "edit" do
    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          expect(Clue::ChatProperty).to receive(:find).once
          get :edit, params: { id: chat_property_id }
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          get :edit, params: { id: chat_property_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "update" do
    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Clue::ChatProperty).to receive(:find).and_return(initial_chat_property)
          expect_any_instance_of(Clue::ChatProperty).to receive(:update).once.and_return(double(success?: true))
          allow(History::AddAudit).to receive(:add)
          patch :update, params: { id: chat_property_id, clue_chat_property: {:id => chat_property_id, :r9k_only_chat => "1"} }
        end

        it_behaves_like "it redirects the user and displays a success message"
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Clue::ChatProperty).to receive(:find).and_return(initial_chat_property)
          expect_any_instance_of(Clue::ChatProperty).to receive(:update).once.and_return(double(success?: false))
          patch :update, params: { id: chat_property_id, clue_chat_property: {:id => chat_property_id} }
        end

        it_behaves_like "it redirects the user and displays an error"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          patch :update, params: { id: chat_property_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end
end
