require 'rails_helper'

RSpec.describe Dmca::TakedownsController do
  include_context 'mocked permissions'
  let(:permission) { 'dmca_takedowns' }

  shared_examples_for 'it does not call the backend' do
    it 'does not invoke a service method' do
      expect(Dmca::Takedown).not_to receive(backend_method)
      get action, params: params
      expect(response).to have_http_status(:success)
    end
  end

  shared_examples_for 'it calls the backend' do
    it 'invokes a service method' do
      expect(Dmca::Takedown).to receive(backend_method).once
      get action, params: params
      expect(response).to have_http_status(:success)
    end
  end

  describe 'takedowns_received' do
    include_context 'an authorized user'
    let(:action) { :takedowns_received }
    let(:backend_method) { :find_received }

    context 'without filter parameters' do
      let(:params) { }

      it_behaves_like 'it does not call the backend'
    end

    context 'with filter parameters' do
      let(:params) { { takedown: { user_id: '1234', window_start: '2018-06-25 17:47:01', window_end: '2018-07-02 17:47:01' } } }

      it_behaves_like 'it calls the backend'
    end
  end

  describe 'takedowns_issued' do
    let(:action) { :takedowns_issued }
    let(:backend_method) { :find_issued }
    let(:params) { { takedown: { user_id: '1234', window_start: '2018-06-25 17:47:01', window_end: '2018-07-02 17:47:01' } } }

    context 'with views' do
      include_context 'stubbed spade requests'

      it 'should send a pageview event to spade' do
        expect_any_instance_of(Faraday::Connection).to receive(:post).and_return(:status => 204, :body => "", :headers => {})
        expect(Dmca::Takedown).to receive(backend_method).once.and_return([])
        get action, params: params
      end
    end

    context 'without filter parameters' do
      include_context 'an authorized user'
      let(:params) { }

      it_behaves_like 'it does not call the backend'
    end

    context 'with filter parameters' do
      include_context 'an authorized user'

      it_behaves_like 'it calls the backend'
    end
  end

end
