require "rails_helper"

describe GiftCards::BalanceController do
  include_context "mocked permissions"

  let(:test_twitch_user) { Twitch::User.new(id: 123, login: "foo") }

  let(:permission) { "gift_cards" }

  describe "index" do
    let(:action) { :index }
    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          get action
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          get action
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end

    describe "when the user query is provided" do
      include_context "an authorized user"
      let(:balances) do
        [
          {
            amount: 100,
            currency: "USD",
            exponent: 2,
          }
        ]
      end

      let(:balance_events) do
        [
          {
            date: '2017-09-18T18:00:00Z',
            purchase_order_id: "1",
            gross_amount_usd: 1000,
          },
          {
            date: '2018-09-18T18:00:00Z',
            purchase_order_id: "2",
            gross_amount_usd: 2000,
          }
        ]
      end

      before do
        expect(Twitch::User).to receive(:all).and_return([test_twitch_user])
        expect(TwitchNami::Balance).to receive(:call).and_return(balances)
        expect(Mulan::GetBalanceHistory).to receive(:call).and_return(balance_events)
        get :index, params: { search_query: "123" }
      end

      it_behaves_like "it returns a success response"
      it "sets the balance variable" do
        expect(@controller.view_assigns["balance"]).to eq(balances)
        expect(@controller.view_assigns["balance_history"]).to eq(balance_events)
      end
    end
  end

  describe "modify" do
    let(:action) { :modify }

    context 'for an authorized user' do
      include_context "an authorized user"
      before :each do
        successful_resp = double("response")
        allow(successful_resp).to receive(:failure).and_return(:NONE)

        expect(Twitch::User).to receive(:find).and_return(test_twitch_user)
        expect(TwitchNami::Transact).to receive(:call).and_return(successful_resp)

        get :modify, params: { balance_id: 1234, amount: "1.01", currency: "USD", balance_action: "add_amount" }
      end

      it_behaves_like 'it redirects the user and displays a success message'
      it 'has the correct success message' do
        expect(flash[:success]).to eq('Updated foo balance')
      end
    end
  end

  describe "wallet_status" do
    let(:action) { :wallet_status }

    context 'for an authorized user' do
      include_context "an authorized user"
      before :each do
        successful_resp = double("response")
        allow(successful_resp).to receive(:failure).and_return(:NONE)

        expect(Twitch::User).to receive(:find).and_return(test_twitch_user)
        expect(TwitchNami::WalletStatus).to receive(:call).and_return(successful_resp)

        get :wallet_status, params: { balance_id: 1234, status_action: "active" }
      end

      it_behaves_like 'it redirects the user and displays a success message'
      it 'has the correct success message' do
        expect(flash[:success]).to eq('Updated foo wallet status')
      end
    end
  end
end
