module History
  describe DeletionsController do
    include_context 'mocked permissions'

    TWIRP_ENDPOINT = "#{Settings.history.endpoint}/twirp/code.justin.tv.foundation.history.History"

    describe 'index' do
      let(:action) { :index }
      let(:permission) { 'history_gdpr_deletions' }
      let(:requester) { 'cool_ldap_login' }
      let(:user_id) { 'my-user-id' }
      let(:deletion_time) { "02/08/2019 3:34 PM" }

      context 'with permissions' do
        include_context 'an authorized user'

        it 'sends requester' do
          expect(History::Deletion).to receive(:list_scheduled_deletions)
            .with(ActionController::Parameters.new({requester: requester}).permit(:requester))
            .and_return({ :items => [] })
          get action, params: { search: { requester: requester } }
        end

        it 'populates deletion results' do
          expect(History::Deletion).to receive(:list_scheduled_deletions)
            .with(ActionController::Parameters.new({requester: ""}).permit(:requester))
            .and_return({:items => {}})
          get action, params: {}
          expect(flash[:error]).to be_nil
        end

        it 'redirect user with error message if service request failed' do
          expect(History::Deletion).to receive(:list_scheduled_deletions).and_raise StandardError.new "service failed"
          get action, params: {}
          expect(response).to redirect_to(history_deletions_path)
          expect(flash[:error]).to eql("service failed")
        end

        it 'redirect the user with error if service returns no data for user' do
          expect(History::Deletion).to receive(:list_scheduled_deletions).and_return({:items => {}})
          expect(History::Deletion).to receive(:get_user_stats).with(user_id).and_return({:data => 0})

          get action, params: { search: { deletion_time: "3_years", user_id: user_id } }
          expect(response).to redirect_to(history_deletions_path)
          expect(flash[:error]).to eql("There is no data associated with user #{user_id}")
        end

        it 'gets a row count for a given user_id' do
          stub_request(:post, "#{TWIRP_ENDPOINT}/ListScheduledDeletes").to_return(:status => 200, :body => "", :headers => {})
          expect(History::Deletion).to receive(:list_scheduled_deletions).and_return({:items => {}})
          expect(History::Deletion).to receive(:get_user_stats).and_return({:data => 20})

          get action, params: { search: { deletion_time: "3_years", user_id: user_id } }
        end
      end

      context 'a user without appropriate permissions' do
        include_context 'an unauthorized user'

        it 'redirects the user and displays an error' do
          get action, params: {}
        end
      end
    end

    describe 'create' do
      let(:action) { :create }
      let(:permission) { 'history_gdpr_deletions' }
      let(:requester) { 'cool_ldap_login' }
      let(:user_id) { 'my-user-id' }
      let(:deletion_time_str) { "02/08/2019 3:34 PM" }
      let(:deletion_time) { Google::Protobuf::Timestamp.new(seconds: Time.parse(deletion_time_str).to_i) }

      context 'with permissions' do
        include_context 'an authorized user'

        it 'request a deletion' do
          stub_request(:post, "#{Settings.history.endpoint}/v1/authed/audits").to_return(:status => 200, :body => "", :headers => {})
          expect(History::Deletion).to receive(:request_deletion)
            .with(user_id, requester, deletion_time)
            .and_return({})

          post action, params: {
            user_id: user_id,
            deletion_time: deletion_time_str,
          }
          expect(response).to redirect_to(history_deletions_path)
          expect(flash[:success]).to eql("Successfully requested deletion")
        end

        it 'redirects user with error if deletion time is not specified' do
          post action, params: {
            user_id: user_id,
          }
          expect(response).to redirect_to(history_deletions_path)
          expect(flash[:error]).to eql("Deletion Time required")
        end

        it 'redirect user with error message if service request failed' do
          stub_request(:post, "#{TWIRP_ENDPOINT}/RequestDeletion").to_return(:status => 200, :body => "", :headers => {})
          expect(History::Deletion).to receive(:request_deletion)
            .with(user_id, requester, deletion_time)
            .and_raise StandardError.new "service failed"

          post action, params: {
            user_id: user_id,
            deletion_time: deletion_time_str,
          }
          expect(response).to redirect_to(history_deletions_path)
          expect(flash[:error]).to eql("service failed")
        end

      end
    end
  end
end
