require 'rails_helper'

RSpec.describe Keys::BatchesController do
  include_context 'mocked permissions'

  let(:permission) { 'product_key_generation' }

  let(:test_pool_id) { 'test-pool' }
  let(:test_batch_id) { 'test-batch' }
  let(:test_number_of_keys) { 100 }

  describe 'index' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        get :index, params: { pool_id: test_batch_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      before do
        expect(Keys::Batch).to receive(:all).and_return(double(success?: true, status: 200, body: {'key_batches' => []}))
        get :index, params: { pool_id: test_batch_id}
      end

      it_behaves_like 'it returns a success response'
    end
  end

  describe 'create' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        post :create
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'
      let(:batch) { Keys::Batch.new }

      before do
        expect(Keys::Batch).to receive(:new).and_return(batch)
        expect(batch).to receive(:create).and_return(double(success?: true, status: 200))
        post :create, params: { keys_batch: { pool_id: test_pool_id, batch_id: test_batch_id, number_of_keys: test_number_of_keys } }
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end
  end

  describe 'deactivate' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        post :deactivate, params: { batch_id: test_batch_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      let(:batch) { Keys::Batch.new }

      before do
        expect(Keys::Batch).to receive(:from_attributes).and_return(batch)
        expect(batch).to receive(:update_status).and_return(double(success?: true, status: 200))
        post :deactivate, params: { batch_id: test_batch_id}
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end
  end

  describe 'activate' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        post :activate, params: { batch_id: test_batch_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      let(:batch) { Keys::Batch.new }

      before do
        expect(Keys::Batch).to receive(:from_attributes).and_return(batch)
        expect(batch).to receive(:update_status).and_return(double(success?: true, status: 200))
        post :activate, params: { batch_id: test_batch_id}
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end
  end
end
