require 'rails_helper'

RSpec.describe Keys::PoolsController do
  include_context 'mocked permissions'

  let(:permission) { 'product_key_generation' }

  let(:test_pool_id) { 'test-pool' }
  let(:test_product_type) { 'test-product-type' }
  let(:test_sku) { 'test-sku' }
  let(:test_description) { 'test-description' }
  let(:test_handler) { 'test-handler' }

  describe 'index' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        get :index
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'

      before do
        expect(Keys::Pool).to receive(:all).and_return(double(success?: true, status: 200, body: {'key_pools' => []}))
        get :index
      end

      it_behaves_like 'it returns a success response'
    end
  end

  describe 'create' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        post :create
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'
      let(:pool) { Keys::Pool.new }

      before do
        expect(Keys::Pool).to receive(:new).and_return(pool)
        expect(pool).to receive(:create).and_return(double(success?: true, status: 200))
        post :create, params: { keys_pool: { pool_id: test_pool_id, product_type: test_product_type, sku: test_sku, description: test_description, handler: test_handler} }
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end
  end

  describe 'deactivate' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        post :deactivate, params: { pool_id: test_pool_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'
      let(:pool) { Keys::Pool.new }

      before do
        expect(Keys::Pool).to receive(:from_attributes).and_return(pool)
        expect(pool).to receive(:update_status).and_return(double(success?: true, status: 200))
        post :deactivate, params: { pool_id: test_pool_id}
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end
  end

  describe 'activate' do
    context 'a user without permission' do
      include_context 'an unauthorized user'

      before do
        post :activate, params: { pool_id: test_pool_id}
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'a user with permission' do
      include_context 'an authorized user'
      let(:pool) { Keys::Pool.new }

      before do
        expect(Keys::Pool).to receive(:from_attributes).and_return(pool)
        expect(pool).to receive(:update_status).and_return(double(success?: true, status: 200))
        post :activate, params: { pool_id: test_pool_id}
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end
  end
end
