require "rails_helper"

RSpec.describe Payments::PurchaseProfilesController do
  include_context "mocked permissions"

  let(:permission) { "purchase_profiles" }

  describe "#index" do
    subject { get :index, params: params }

    let(:params) { {} }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"
        include_context 'it creates any tracking call'

        context 'with no params' do
          let(:params) { {} }

          before do
            allow(Payments::PurchaseProfile).to receive(:filter).and_return([])

            subject
          end

          it_behaves_like "it returns a success response"
        end

        context "with purchaser_id" do
          let(:purchaser_login) { 'cool_user' }
          let(:params) { { purchaser_login: purchaser_login } }
          let(:user) { Twitch::User.new(id: 12345, login: purchaser_login) }
          let(:resp) { [] }

          before do
            allow(Twitch::User).to receive(:find_by_login).with(purchaser_login).and_return(user)
            allow(Payments::PurchaseProfile).to receive(:filter).and_return(resp)
          end

          context "with no such user" do
            before do
              allow(Twitch::User).to receive(:find_by_login).with(purchaser_login).and_return(nil)
              subject
            end

            it_behaves_like "it redirects the user and displays an error"
          end

          context 'with no items returned' do
            before { subject }
            it_behaves_like "it returns a success response"
          end

          context 'with items returned' do
            let(:resp) { [double(:purchase_profile_stub, load_associations: [])] }
            before { subject }
            it_behaves_like "it returns a success response"
          end
        end
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          subject
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#show" do
    let(:purchase_profile_id) { 12345 }

    describe "permissions" do
      let(:purchase_profile) do
        double(:purchase_profile,
                blank?: false,
                state: 'migrated_v2')
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Payments::PurchaseProfile).to receive(:find).once
            .with(purchase_profile_id.to_s).and_return(purchase_profile)

          get :show, params: { id: purchase_profile_id }
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          get :show, params: { id: purchase_profile_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#cancel" do
    let(:purchase_profile_id) { 12345 }

    describe "permissions" do

      let(:purchase_profile) do
        double(:purchase_profile,
                blank?: false,
                state: 'migrated_v2')
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"
        include_context 'it creates any tracking call'

        before do
          allow(Payments::PurchaseProfile).to receive(:cancel).once
            .with(purchase_profile_id.to_s).and_return(true)
          post :cancel, params: { purchase_profile_id: purchase_profile_id }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :cancel, params: { purchase_profile_id: purchase_profile_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#do_not_renew" do
    let(:purchase_profile_id) { 12345 }

    describe "permissions" do

      let(:purchase_profile) do
        double(:purchase_profile,
                blank?: false,
                state: 'migrated_v2')
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"
        include_context 'it creates any tracking call'

        before do
          allow(Payments::PurchaseProfile).to receive(:do_not_renew).once
            .with(purchase_profile_id.to_s).and_return(true)
          post :do_not_renew, params: { purchase_profile_id: purchase_profile_id }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :do_not_renew, params: { purchase_profile_id: purchase_profile_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#sync_ticket_access" do
    let(:purchase_profile_id) { 12345 }

    describe "permissions" do

      let(:purchase_profile) do
        double(:purchase_profile,
                blank?: false,
                state: 'migrated_v2')
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"
        include_context 'it creates any tracking call'

        before do
          allow(Payments::SyncTicketAccess).to receive(:do).once
            .with(purchase_profile_id.to_s).and_return(true)
          post :sync_ticket_access, params: { purchase_profile_id: purchase_profile_id }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :sync_ticket_access, params: { purchase_profile_id: purchase_profile_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end
end
