# frozen_string_literal: true

require 'rails_helper'

RSpec.describe Staff::MembersController do
  include_context 'mocked permissions'

  let(:permission) { 'staff_management' }
  let(:test_user_id) { '123' }
  let(:test_twitch_user) { Twitch::User.new(id: test_user_id) }

  describe 'index' do
    let(:action) { :index }

    context 'with permissions' do
      include_context 'an authorized user'

      it 'returns a success response' do
        get action
      end

      it 'redirects if id is set' do
        res = get action, params: { id: 'derp' }
        expect(res.status).to eq(302)
      end
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user'

      it 'redirects the user and displays an error' do
        get action, params: { id: 'derp' }
      end
    end
  end

  describe 'show' do
    let(:action) { :show }
    let(:id) { 'my-member-id' }

    shared_examples_for 'it loads a member' do
      it_behaves_like 'it returns a success response'

      it 'sets member' do
        expect(@controller.view_assigns['member']).to eq(member)
      end
    end

    context 'with permissions' do
      include_context 'an authorized user'

      before(:each) do
        allow(Staff::Member).to receive(:find).with(id).and_return(member)
        get action, params: { id: id }
      end

      context 'if member is found' do
        let(:member) { Staff::Member.new }

        it_behaves_like 'it loads a member'
      end

      context 'if a member is not found' do
        let(:member) { Staff::Member.from_errors(['not found']) }

        it_behaves_like 'it loads a member'

        it 'flashes an error' do
          expect(flash[:error]).to be_present
          expect(flash[:warning]).to be_present
        end
      end
    end
  end
end
