require "rails_helper"

RSpec.describe Subscriptions::BadgesController do
  include_context "mocked permissions"

  let(:channel_id) { 12345 }

  describe "#index" do
    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        let(:permission) { ["ticket_products", "ticket_products"] }

        before do
          allow(Subscriptions::Badge).to receive(:existing_and_available_for).once
            .with(channel_id: channel_id.to_s).and_return([])

          allow(Subscriptions::BadgeTimeoutStatus).to receive(:find_by).once
            .with(channel_id: channel_id.to_s).and_return(double(persisted?: true))

          get :index, params: { channel_id: channel_id }
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        let(:permission) { "ticket_products" }

        before do
          get :index, params: { channel_id: channel_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#create" do
    describe "permissions" do
      let(:permission) { "ticket_products" }

      let(:post_form_data) do
        {
          image_1x: "1x",
          image_2x: "2x",
          image_4x: "4x",
          required_tenure_months: 12
        }
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"

        let(:badge) { Subscriptions::Badge.new }

        before do
          expect_any_instance_of(Subscriptions::Badge).to receive(:save).once.and_return(true)

          post :create, params: { channel_id: channel_id, subscriptions_badge: post_form_data }
        end

        it_behaves_like "it redirects the user and displays a success message"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :create, params: { channel_id: channel_id, subscriptions_badge: post_form_data }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#destroy" do
    let(:badge_id) { 9999 }

    describe "permissions" do
      let(:permission) { "ticket_products" }

      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          expect_any_instance_of(Subscriptions::Badge).to receive(:destroy).once.and_return(true)

          delete :destroy, params: { channel_id: channel_id, id: badge_id }
        end

        it_behaves_like "it redirects the user and displays a success message"
      end

      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          expect_any_instance_of(Subscriptions::Badge).to receive(:destroy).once.and_return(true)

          delete :destroy, params: {
            channel_id: channel_id,
            channel_tenure_method_id: "#{channel_id}:1:CUMULATIVE",
            id: badge_id,
          }
        end

        it_behaves_like "it redirects the user and displays a success message"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          delete :destroy, params: { channel_id: channel_id, id: badge_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end
end
