require 'rails_helper'

RSpec.describe Subscriptions::ResubTokensController do
  include_context 'mocked permissions'

  let(:permission) { 'ticket_products' }

  describe 'allow_replay' do
    describe 'permissions' do
      let(:action) { :allow_replay }
      let(:userID) { '2' }
      let(:channelID) { '1' }
      let(:tenure) { '5' }

      let(:twirp_allow_replay_response) do
        data = Code::Justin::Tv::Revenue::Subscriptions::AllowReplayForLastUsedChatNotificationTokenResponse.new
        Twirp::ClientResp.new(data, nil)
      end

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'

        before do
          expect(Substwirp::ResubTokens).to receive(:allow_replay).with(
            userID,
                channelID,
                tenure
          ).and_return(twirp_allow_replay_response)

          post action, params: {
              owner_id: userID,
              channel_id: channelID,
              tenure: tenure
          }
        end
      end

      it 'redirects to resub tokens index view after deleting grant' do
        expect(response.code).to eq('200')
      end
    end
  end

  describe 'index' do
    describe 'permissions' do
      let(:action) { :index }
      let(:channelID) { '1' }
      let(:channelName) { 'name' }
      let(:userID) { '2' }
      let(:userName) { 'anotherName' }
      let(:used_tokens) do
        [
          {
                  owner: "193141706",
                  channel: "193141706",
                  tenure: "5",
                  method: "cumulative",
                  streak_shared: true,
                  streak_value: 5,
                  custom_message_text: "woo 9 months!",
                  custom_message_shared: true,
                  product: "1212026",
                  tier: "3000",
                  updated_at: {
                      seconds: -62_135_596_800,
                      nanos: 0
                  },
                  created_at: {
                      seconds: -62_135_596_800,
                      nanos: 0
                  }
              }
        ]
      end
      let(:channel) do
        Twitch::User.new(
          id: channelID,
          login: channelName
        )
      end
      let(:user) do
        Twitch::User.new(
          id: userID,
          login: userName
        )
      end
      let(:twirp_available_token_response) do
        Code::Justin::Tv::Revenue::Subscriptions::GetUserChannelChatNotificationTokenResponse.new(
          chat_notification_token: {
                  id: '1234'
              }
        )
      end

      let(:twirp_token_history_response) do
        data = Code::Justin::Tv::Revenue::Subscriptions::GetAllUsedChatNotificationTokensResponse.new used_chat_tokens: used_tokens
        Twirp::ClientResp.new(data, nil)
      end

      context 'a user with appropriate permissions' do
        include_context 'an authorized user'
        include_context 'it creates any tracking call'

        before do
          expect(Twitch::User).to receive(:find).with(channelID).and_return(channel)
          expect(Twitch::User).to receive(:find).with(userID).and_return(user)
          expect(Substwirp::ResubTokens).to receive(:get_available_token).with(
            userID,
              channelID
          ).and_return(twirp_available_token_response)
          expect(Substwirp::ResubTokens).to receive(:get_all_used_tokens).with(
            userID,
              channelID
          ).and_return(twirp_token_history_response)
          get action, params: {
              resub_token: {
              channelID: channelID,
              channelName: nil,
              userID: userID,
              userName: nil
              }
          }
        end

        it 'redirects to new tenure index view when a channel/user cannot be found' do
          expect(response.code).to eq('200')
        end
      end
    end
  end
end
