require "rails_helper"

RSpec.describe Subscriptions::TicketProductRevenueConfigsController do
  include_context "mocked permissions"

  let(:permission) { "ticket_products" }

  describe "#index" do
    let(:ticket_product_id) { 123 }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Substwirp::RevenueConfig).to receive(:get_configs_for_product_id).once
            .with(ticket_product_id.to_s).and_return([])

          get :index, params: { id: ticket_product_id }
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          get :index, params: { id: ticket_product_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#edit" do
    let(:revenue_config_id) { 12345 }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Substwirp::RevenueConfig).to receive(:get_by_id).once
            .with(revenue_config_id.to_s).and_return(double(persisted?: true))

          get :edit, params: { id: revenue_config_id }
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          get :edit, params: { id: revenue_config_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#new" do
    let(:ticket_product_id) { 1234 }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          get :new, params: { ticket_product_id: ticket_product_id }
        end

        it_behaves_like "it returns a success response"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          get :new, params: { ticket_product_id: ticket_product_id }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end
  end

  describe "#update" do
    let(:id) { 12345 }
    let(:start_date) { '2019-01-01T00:00:00.000Z' }
    let(:rev_split_percent) { 'rev_split_percent' }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Substwirp::RevenueConfig).to receive(:update).once
            .with(id.to_s, start_date, rev_split_percent).and_return(double(persisted?: true))
          expect(Substwirp::RevenueConfig).to receive(:update).once

          post :update, params: {
            id: id,
            substwirp_revenue_config: {
              id: id,
              start_date: start_date,
              rev_split_percent: rev_split_percent
            }
          }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :update, params: {
            id: id,
            substwirp_revenue_config: {
              id: id,
              start_date: start_date,
              rev_split_percent: rev_split_percent,
            }
          }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end

    describe "failed service call" do
      include_context "an authorized user"

      before do
        allow(Substwirp::RevenueConfig).to receive(:update).once
          .with(id.to_s, start_date, rev_split_percent).and_raise("service error")
        expect(Substwirp::RevenueConfig).to receive(:update).once

        post :update, params: {
          id: id,
          substwirp_revenue_config: {
            id: id,
            start_date: start_date,
            rev_split_percent: rev_split_percent
          }
        }
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe "#create" do
    let(:id) { 12345 }
    let(:product_id) { 123 }
    let(:start_date) { '2019-01-01T00:00:00.000Z' }
    let(:rev_split_percent) { 'rev_split_percent' }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Substwirp::RevenueConfig).to receive(:create).once
            .with(product_id.to_s, start_date, rev_split_percent).and_return(double(success?: true))
          expect(Substwirp::RevenueConfig).to receive(:create).once

          post :create, params: {
            substwirp_revenue_config: {
              product_id: product_id,
              start_date: start_date,
              rev_split_percent: rev_split_percent
            }
          }
        end

        it_behaves_like 'it redirects the user and does not display an error'
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :create, params: {
            substwirp_revenue_config: {
              product_id: product_id,
              start_date: start_date,
              rev_split_percent: rev_split_percent,
            }
          }
        end

        it_behaves_like "it redirects the user and displays an error"
      end
    end

    describe "failed service call" do
      include_context "an authorized user"

      before do
        allow(Substwirp::RevenueConfig).to receive(:create).once
          .with(product_id.to_s, start_date, rev_split_percent).and_raise("service error")
        expect(Substwirp::RevenueConfig).to receive(:create).once

        post :create, params: {
          substwirp_revenue_config: {
            product_id: product_id,
            start_date: start_date,
            rev_split_percent: rev_split_percent
          }
        }
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe "#delete" do
    let(:id) { 12345 }

    describe "permissions" do
      context "a user with appropriate permissions" do
        include_context "an authorized user"

        before do
          allow(Substwirp::RevenueConfig).to receive(:delete).once
            .with(id.to_s).and_return(double(success?: true))
          expect(Substwirp::RevenueConfig).to receive(:delete).once

          post :delete, params: { ticket_product_revenue_config_id: id }
        end

        it_behaves_like "it redirects the user and does not display an error"
      end

      context "a user without appropriate permissions" do
        include_context "an unauthorized user"

        before do
          post :delete, params: { ticket_product_revenue_config_id: id }
        end

        it_behaves_like 'it redirects the user and displays an error'
      end
    end

    describe "failed service call" do
      include_context "an authorized user"

      before do
        allow(Substwirp::RevenueConfig).to receive(:delete).once
          .with(id.to_s).and_raise("service error")
        expect(Substwirp::RevenueConfig).to receive(:delete).once

        post :delete, params: { ticket_product_revenue_config_id: id }
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end
end
