require 'rails_helper'

RSpec.describe Twitch::ImpersonatesController do
  include_context 'mocked permissions'

  let(:permission) { 'impersonate' }

  describe 'create' do
    let(:user_id) { 'my-user-id' }
    let(:user) { Twitch::User.new }
    let(:action) { :create }
    let(:reason) { 'my-reason' }
    let(:ticket_id) { 'my-ticket-id' }
    let(:params) do
      {
        user_id: user_id,
        ticket_id: ticket_id,
        reason: reason
      }
    end
    let(:user_url) { 'https://my-user-url' }
    let(:audit) do
      History::Audit.new(
        action: 'impersonate_user',
        user_type: History::AUDIT_LDAP_USER,
        user_id: 'cool_ldap_login',
        resource_type: History::AUDIT_TWITCH_USER,
        resource_id: user_id,
        description: "IP address 0.0.0.0"
      )
    end
    let(:user_impersonation) { Passport::UserImpersonation.new }

    before(:each) do
      expect(Twitch::User).to receive(:find).with(user_id).and_return(user)
    end

    context 'with permissions' do
      include_context 'an authorized user acting on a twitch user'

      context 'with enough fields' do
        include_context 'it creates any tracking call'

        before(:each) do
          expect(Passport::UserImpersonation).to receive(:new)
            .with(
              user_id: user_id,
              user_ip: '0.0.0.0',
              admin_id: 'cool_ldap_login',
              ticket_id: ticket_id,
              reason: reason
            )
            .and_return(user_impersonation)
          expect(user_impersonation).to receive(:audit_impersonation)
          expect(user_impersonation).to receive(:url)
            .and_return(user_url)

          response = get action, params: params
          expect(response.header['Location'])
            .to eq(user_url)
          expect(response.status)
            .to eq(302)
        end

        it 'redirects the user to passport' do
        end

        context 'with only reason' do
          let(:ticket_id) { '' }

          it 'redirects the user to passport' do
          end
        end
      end

      context 'with invalid parameters' do
        before(:each) do
          get action, params: params
          expect(response.status)
            .to eq(302)
          expect(flash[:error]).to be_present
        end

        context 'with no reason' do
          let(:reason) { '' }

          it 'redirects the user back' do
          end
        end

        context 'reason too long' do
          let(:reason) { 'too long ' * 1000 }

          it 'redirects the user back' do
          end
        end

        context 'ticket_id too long' do
          let(:ticket_id) { 'too long ' * 1000 }

          it 'redirects the user back' do
          end
        end
      end
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      it 'redirects the user and displays an error' do
        get action, params: params
      end
    end
  end
end
