require 'rails_helper'
require_relative './shared_contexts'

describe Twitch::PasswordsController do
  include_context 'mocked permissions'
  include_context 'with example twitch users'

  let(:permission) { any_permission_in('users', 'twitch_user:update_details') }
  let(:user) { twitch_user }

  before(:each) do
    expect(Twitch::User)
      .to receive(:find)
      .with(twitch_user_id)
      .and_return(twitch_user)
  end

  describe 'change_password' do
    let(:action) { :change_password }
    let(:new_password) { 'my-new-password' }
    let(:params) { { user_id: twitch_user_id, new_password: new_password } }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user acting on a twitch user'
      include_context 'it creates any tracking call'

      before(:each) do
        expect(twitch_user)
          .to receive(:change_password)
          .with(new_password)
          .and_return(true)
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'force_reset' do
    let(:action) { :force_reset }
    let(:params) { { user_id: twitch_user_id } }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user acting on a twitch user'
      include_context 'it creates any tracking call'

      before(:each) do
        expect(twitch_user).to receive(:force_password_reset).and_return(true)
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end
end
