require 'rails_helper'

RSpec.describe Twitch::SuspensionsController do
  include_context 'mocked permissions'

  let(:permission) { 'support_tools' }

  describe 'unban a user as' do
    let(:user) { Twitch::User.new(displayname: "foo", id: "1234") }

    context 'an admin with appropriate permissions' do
      include_context 'an authorized user'

      before do
        allow(Twitch::User).to receive(:find).with(user.id).and_return(user)
        expected_options = {
          "reporter" => "cool_ldap_login",
          "description" => "unban foo id: 1234 by cool_ldap_login",
          :decrement => 0,
          :types => []
        }
        allow(user).to receive(:tos_unban).with(expected_options).and_return(true)
        delete :destroy, params: { user_id: user.id }
      end

      it_behaves_like 'it redirects the user and does not display an error'
    end

    context 'an admin with appropriate permission but unban fails' do
      include_context 'an authorized user'

      before do
        allow(Twitch::User).to receive(:find).with(user.id).and_return(user)
        allow(user).to receive(:tos_unban).and_return(false)
        delete :destroy, params: { user_id: user.id }
      end

      it_behaves_like 'it redirects the user and displays an error'
    end

    context 'an admin without appropriate permissions' do
      include_context 'an unauthorized user'

      before { delete :destroy, params: { user_id: user.id } }

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'unban a user with optional params' do
    tos_count = 3
    dmca_count = 2
    let(:user) { Twitch::User.new(displayname: "foo", id: "1234", tos_violation_count: tos_count, dmca_violation_count: dmca_count) }
    let(:audit) do
      History::Audit.new(
        action: "unban_user",
        user_type: "ldap_user",
        user_id: "cool_ldap_login",
        resource_type: "twitch_user",
        resource_id: "1234",
        description: "unban foo id: 1234 by cool_ldap_login",
        changes: [
          {
            attribute: "tos_violation_count",
            old_value: tos_count,
            new_value: 0
          }, {
            attribute: "dmca_violation_count",
            old_value: dmca_count,
            new_value: 0
          }
        ]
      )
    end
    let(:tos_query) { {:decrement => 3, :ban_type => "tos"}.to_query }
    let(:dmca_query) { {:ban_type => "dmca"}.to_query }
    let(:expected_url) {"/users/1234/ban?#{tos_query}&#{dmca_query}"}
    let(:usersservice_response) { double "yay" }

    context 'an admin with appropriate permissions' do
      include_context 'an authorized user'

      before do
        allow(Twitch::User).to receive(:find).with(user.id).and_return(user)
        allow(user).to receive(:fetch_id).and_return("1234")

        expect(Twitch::User).to receive(:delete).with(expected_url).and_return(usersservice_response)
        expect(usersservice_response).to receive(:success?).and_return(true).at_least(1)

        expect(History::AddAudit).to receive(:add).with(audit)

        delete :destroy, params: { user_id: user.id, tos: "true", dmca: "true", decrement: "3" }
      end
      it_behaves_like 'it redirects the user and does not display an error'
    end
  end
end