require 'rails_helper'
require_relative './shared_contexts'

describe Twitch::UserRenamesController do
  include_context 'mocked permissions'
  include_context 'with example twitch users'

  let(:permission) { 'user_rename' }
  let(:user) { twitch_user }
  let(:params) { { user_id: twitch_user_id } }

  before(:each) do
    expect(Twitch::User)
      .to receive(:find)
      .with(twitch_user_id)
      .and_return(twitch_user)
  end

  describe 'new' do
    let(:action) { :new }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it returns a success response'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'create' do
    let(:action) { :create }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user acting on a twitch user'

      before(:each) do
        expect(twitch_user)
          .to receive(:rename)
          .with(
            ActionController::Parameters.new(
              params.merge(
                ldap_login: ldap_name,
                controller: 'twitch/user_renames',
                action: 'create'
              )
            )
          )
          .and_return(true)
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'new_override' do
    let(:action) { :new_override }
    let(:rename_error) { 'my rename error' }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user acting on a twitch user'

      before(:each) do
        flash[:rename_error] = rename_error
        post action, params: params
      end

      it_behaves_like 'it returns a success response'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end

  describe 'create_override' do
    let(:action) { :create_override }

    context 'a user with appropriate permissions' do
      include_context 'an authorized user acting on a twitch user'

      before(:each) do
        expect(twitch_user)
          .to receive(:rename)
          .with(
            ActionController::Parameters.new(
              params.merge(
                ldap_login: ldap_name,
                controller: 'twitch/user_renames',
                action: 'create_override',
                admin_override: true
              )
            )
          )
          .and_return(true)
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays a success message'
    end

    context 'a user without appropriate permissions' do
      include_context 'an unauthorized user acting on a twitch user'

      before(:each) do
        post action, params: params
      end

      it_behaves_like 'it redirects the user and displays an error'
    end
  end
end
