require 'rails_helper'

module Twitch
  RSpec.describe ValidatedNonPartnerController do
    let(:user) { User.new(id: 1234) }
    let(:permission) { 'manage_validated_non_partner' }
    include_context 'mocked permissions'

    before do
      allow(User).to receive(:find).with('1234').and_return(user)
    end

    describe '#edit' do
      include_context 'an authorized user acting on a twitch user'

      it 'sets the user' do
        get :edit, params: { user_id: '1234' }
        expect(@controller.view_assigns['user']).to eq(user)
      end
    end

    describe 'removing validated non partner' do
      let(:data) { Code::Justin::Tv::Revenue::Ripley::EditValidatedNonPartnerResponse.new }
      include_context 'an authorized user acting on a twitch user'

      before do
        allow(Ripley::ValidatedNonPartner).to receive(:delete).and_return(Twirp::ClientResp.new(data, nil))
        put :update, params: { user_id: '1234', validated_non_partner: "false" }
      end

      it 'deletes user from validated non partner' do
        expect(Ripley::ValidatedNonPartner).to have_received(:delete).with(
          '1234'
        )
      end

      it 'redirects to the user show page' do
        expect(response).to redirect_to '/twitch/users/1234'
      end

      it 'shows a flash success message' do
        expect(flash[:success]).to eq 'Successfully updated validated non partner to false'
      end

    end

    describe 'adding validated non partner' do
      let(:data) { Code::Justin::Tv::Revenue::Ripley::EditValidatedNonPartnerResponse.new }
      include_context 'an authorized user acting on a twitch user'

      before do
        allow(Ripley::ValidatedNonPartner).to receive(:add).and_return(Twirp::ClientResp.new(data, nil))
        put :update, params: { user_id: '1234', validated_non_partner: "true" }
      end

      it 'adds user to validated non partner' do
        expect(Ripley::ValidatedNonPartner).to have_received(:add).with(
          '1234'
        )
      end

      it 'redirects to the user show page' do
        expect(response).to redirect_to '/twitch/users/1234'
      end

      it 'shows a flash success message' do
        expect(flash[:success]).to eq 'Successfully updated validated non partner to true'
      end
    end
  end
end
